<?php

namespace Drupal\banking_accounting;

use Drupal\banking\Entity\BankTransaction;
use Drupal\banking_ogm\Entity\BankingOGM;

use Drupal\accounting\Entity\AccountingAccount;
use Drupal\accounting\Entity\AccountingJournal;
use Drupal\accounting\BookEntryServiceInterface;

/**
 * Class BankingAccountingService.
 */
class BankingAccountingService implements BankingAccountingServiceInterface {

  /**
   * Drupal\accounting\BookEntryServiceInterface definition.
   *
   * @var \Drupal\accounting\BookEntryServiceInterface
   */
  protected $accountingBookEntry;

  /**
   * Constructs a new BankingAccountingService object.
   */
  public function __construct(BookEntryServiceInterface $accounting_book_entry) {
    $this->accountingBookEntry = $accounting_book_entry;
  }

  /**
   * {@inheritdoc}
   */
  public function matchingAccountsForTransaction(BankTransaction $bankTransaction) {
    $args = [
      'account_ids' => []
    ];
    \Drupal::moduleHandler()->alter('bank_transaction_accounts', $bankTransaction, $args);

    if (count($args['account_ids']) > 0) {
      return AccountingAccount::loadMultiple($args['account_ids']);
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function ogmsForAccount(AccountingAccount $account) {
    $ids = \Drupal::entityQuery('banking_ogm')
      ->condition('account', $account->id())
      ->execute();

    return BankingOGM::loadMultiple($ids);
  }

  /**
   * {@inheritdoc}
   */
  public function bookBankTransaction(BankTransaction $bankTransaction, AccountingAccount $account) {
    if (count(
        \Drupal::entityQuery('journal_entry')
          ->condition('field_bank_transaction', $bankTransaction->id())
          ->execute())) {
      return NULL;
    }

    $bank_account_id = $bankTransaction->getBankAccount()->get('field_account')->target_id;
    $bank_journal_id = $bankTransaction->getBankAccount()->get('field_journal')->target_id;

    $entry = $this->accountingBookEntry->bookEntry(
      'bank',
      $bankTransaction->getDate(),
      AccountingJournal::load($bank_journal_id),
      [
        [
          'name' => $bankTransaction->getName(),
          'amount' => $bankTransaction->getAmount(),
          'account_id' => $bank_account_id,
        ],
        [
          'name' => $bankTransaction->getName(),
          'amount' => bcmul($bankTransaction->getAmount(), -1, 2),
          'account_id' => $account->id(),
        ]
      ]
    );
    $entry->set('field_bank_transaction', $bankTransaction);
    $entry->save();

    $bankTransaction->setState('done');
    $bankTransaction->save();
    return $bankTransaction;
  }
}
