<?php

namespace Drupal\banking_export\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\Core\Link;

use Drupal\banking_export\Entity\PaymentOrder;

use AbcAeffchen\Sephpa\SephpaCreditTransfer;
use AbcAeffchen\SepaUtilities\SepaUtilities;

/**
 * Class CreateSepaSctForm.
 */
class CreateSepaSctForm extends FormBase {


  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'create_sepa_sct_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['payment_order'] = [
      '#type' => 'entity_autocomplete',
      '#title' => $this->t('Payment Order'),
      '#weight' => '0',
      '#target_type' => 'payment_order',
      '#required' => TRUE,
    ];
    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $order = PaymentOrder::load($form_state->getValue('payment_order'));

    $creditTransferFile = new SephpaCreditTransfer(
      'Initiator Name',
      "{$order->getName()}",
      SephpaCreditTransfer::SEPA_PAIN_001_001_03,
      [
        // Required information about the debtor.
        // ID of the payment collection.
        'pmtInfId' => "{$order->getName()}",
        // Debtor (max 70 characters).
        'dbtr' => $order->get('bank_account')->entity->get('name')->value,
        // IBAN of the Debtor.
        'iban' => preg_replace('/\s+/', '', $order->get('bank_account')->entity->get('number')->value),
        'ccy' => 'EUR',
        // BatchBooking, only 'true' or 'false'.
        'btchBookg' => 'false',
        // Requested Execution Date: YYYY-MM-DD.
        /* 'reqdExctnDt' => '2013-11-25', */
        // Just an information, this do not affect the payment (max 70 char).
        /* 'ultmtDebtr' => 'Ultimate Debtor Name'  */
      ]
    );

    foreach ($order->get('lines')->referencedEntities() as $line) {
      $account_number = preg_replace('/\s+/', '', $line->get('account_number')->value);

      if ($line->get('amount')->value < 0.01) {
        continue;
      }
          
      if (!$account_number || !SepaUtilities::checkIban($account_number)) {
        drupal_set_message("Unknown or invalid account number '{$account_number}' for line '{$line->getName()}'!", 'error');
        // $account_number = preg_replace('/\s+/', '', $order->get('bank_account')->entity->get('number')->value);
        continue;
      }

      $pmtId = $line->getData('pmtId');

      $creditTransferFile->addPayment([
        // Required information about the creditor.
        // ID of the payment (EndToEndId).
        'pmtId' => $pmtId ? $pmtId : $line->getName(),
        'instdAmt' => $line->get('amount')->value,
        'iban' => $account_number,
        // Creditor (max 70 characters)
        'cdtr' => $line->get('accountholder')->value,
        // Unstructured information about the remittance (max 140 characters).
        'rmtInf' => $line->getName(),
      ]);
    }

    $options = [];
    $data = $creditTransferFile->generateOutput($options);
    $output = file_save_data($data['data'], 'private://payment_order.xml');

    drupal_set_message(t(
        'Exported payment order @url',
        [
          '@url' => Link::fromTextAndUrl(
            'payment_order.xml',
            Url::fromUri(
              file_create_url(
                $output->getFileUri())
            ))->toString(),
        ]
    ));

  }

}
