<?php

namespace Drupal\banking_import\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;

use Drupal\banking_import\Exception\BankingImportException;

/**
 * Class CAMTImportForm.
 */
class CAMTImportForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public static function finishImport($success, $results, $operations) {
    drupal_set_message(t(
        'Imported @number transaction from CAMT File.',
        [
          '@number' => $results['count'],
        ]
    ));
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'camt_import_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $format_manager = \Drupal::service('plugin.manager.banking_import_format_definitions');
    $format_options = array_column($format_manager->getDefinitions(), 'label', 'id');

    $form['file'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('File'),
      '#description' => $this->t('File to import'),
      '#required' => TRUE,
      '#upload_validators' => [
        'file_validate_extensions' => ['xml xda bc2 cod'],
      ],
    ];
    $form['format'] = [
      '#type' => 'select',
      '#title' => $this->t('File format'),
      '#options' => $format_options,
      '#weight' => '0',
      '#required' => TRUE,
    ];
    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    if (!$form_state->getValue('file') || !$form_state->getValue('format')) {
      // Already validated in parent due to required.
      return;
    }

    $definition = \Drupal::service('plugin.manager.banking_import_format_definitions')
      ->createInstance($form_state->getValue('format'), []);
    $file = file_load(reset($form_state->getValue('file')));

    try {
      $definition->validateFile(drupal_realpath($file->GetFileUri()));
    }
    catch (BankingImportException $e) {
      $form_state->setErrorByName(
        'file',
        $e->getMessage());
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $definition = \Drupal::service('plugin.manager.banking_import_format_definitions')
      ->createInstance($form_state->getValue('format'), []);
    $file = file_load(reset($form_state->getValue('file')));
    $batch = $definition->importFileBatch($file);

    $batch['finished'] = '\Drupal\banking_import\Form\CAMTImportForm::finishImport';
    batch_set($batch);

    $form_state->setRedirect('entity.bank_transaction.collection');
  }
}
