<?php
namespace Drupal\banking_import\Plugin\BankingImportFormat;

use Drupal\banking_import\Exception\BankingImportException;
use Drupal\banking_import\Plugin\BankingImportFormatDefinitionBase;

use Money\Currencies\ISOCurrencies;
use Money\Formatter\DecimalMoneyFormatter;
use BadMethodCallException;

use Genkgo\Camt\Config;
use Genkgo\Camt\Reader;

use Genkgo\Camt\DTO\Creditor;
use Genkgo\Camt\DTO\Debtor;

use Drupal\banking\Entity\BankTransaction;
use Drupal\banking\Entity\BankAccount;

/**
 * Prorata.
 *
 * @BankingImportFormatDefinition(
 *   id = "camt053",
 *   label = @Translation("CAMT053"),
 *  )
 */
class CAMT053 extends BankingImportFormatDefinitionBase {

  /**
   * {@inheritdoc}
   */
  public static function objectToArray($object, $seen = []) {
    if (!is_object($object) && !is_array($object)) {
      return $object;
    }
    return array_filter(
      array_map(
        function ($o) use ($seen) {
          $hash = FALSE;
          if (is_object($o)) {
            $hash = spl_object_hash($o);
          }
          if (!$hash || !in_array($hash, $seen)) {
            return CAMT053::objectToArray($o, array_merge($seen, [$hash]));
          }
          return FALSE;
        },
        (array) $object
      )
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function formatAddress($address) {
    if (!$address) {
      return '';
    }

    $comp = $address->getAddressLines() + [
      $address->getStreetName(),
      $address->getTownName(),
      $address->getCountry(),
    ];
    return implode(
      ', ',
      array_filter(
        array_map(
          function ($s)
          {
            return preg_replace('/\s+/', ' ', $s);
          },
          $comp
        )
      )
    );
  }

  /**
   * {@inheritdoc}
   */
  public function validateFile($camt_file) {
    $reader = new Reader(Config::getDefault());
    $message = $reader->readFile($camt_file);

    $accounts = [];
    foreach ($message->getRecords() as $statement) {
      $account = $statement->getAccount()->getIban();
      if (!in_array($account, $accounts)) {
        $accounts[] = $account;
      }
    }

    foreach ($accounts as $account) {
      if (!BankAccount::loadByAccountNumber($account)) {
        throw new BankingImportException(t(
            'This CAMT file contains statements for unknown bank account @account',
            [
              '@account' => $account,
            ]
        ));
      }
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function importFileBatch($file) {
    $camt_file = drupal_realpath($file->GetFileUri());
    $reader = new Reader(Config::getDefault());
    $message = $reader->readFile($camt_file);

    $batch = array(
      'title' => t('Importing CAMT Statement'),
      'operations' => [],
    );

    foreach ($message->getRecords() as $statement) {

      $bank_account = BankAccount::loadByAccountNumber($statement->getAccount()->getIban());
      $statement_from_date = $statement->getFromDate()->format('Y-m-d');
      $statement_to_date = $statement->getToDate()->format('Y-m-d');
      $query = \Drupal::entityQuery('bank_transaction')
        ->condition('date', $statement_from_date, '>=')
        ->condition('date', $statement_to_date, '<=')
        ->condition('bank_account_id', $bank_account->id());
      $values = $query->execute();
      $dates = [];
      foreach (BankTransaction::loadMultiple(array_keys($values)) as $transactie) {
        if (!in_array($transactie->getDate(), $dates)) {
          $dates[] = $transactie->getDate();
        }
      }
      $batch['operations'][] = [
        '\Drupal\banking_import\Plugin\BankingImportFormat\CAMT053::importStatement',
        [
          $statement,
          $dates,
          $file->id(),
        ]
      ];
    }

    return $batch;
  }

  /**
   * {@inheritdoc}
   */
  public static function importStatement($statement, $dates, $file_id, &$context) {
    $currencies = new ISOCurrencies();
    $moneyFormatter = new DecimalMoneyFormatter($currencies);
    $bank_account = BankAccount::loadByAccountNumber($statement->getAccount()->getIban());
    $n=0;

    foreach ($statement->getEntries() as $batch_entry) {
      if (in_array($batch_entry->getBookingDate()->format('Y-m-d'), $dates)) {
        drupal_set_message(
          t("Not importing entry for day @day and account @account: there are already transactions for this day!",
            [
              '@day' => $batch_entry->getBookingDate()->format('Y-m-d'),
              '@account' => $statement->getAccount()->getIban(),
            ]),
          'warning');
        continue;
      }

      foreach ($batch_entry->getTransactionDetails() as $entry) {
        $amount = $entry->getAmountDetails()->getAmount();

        $transaction_data = [
          'name' => $statement->getId() . '/' . $n,
          'bank_account_id' => $bank_account,
          'amount' => $moneyFormatter->format($amount),
          'date' => $batch_entry->getBookingDate()->format('Y-m-d'),
          'import_file' => $file_id,
          'extra' => [
            'camt-transaction' => CAMT053::objectToArray($entry),
          ],
        ];

        $information = [];

        // TRIODOS
        try {
          $information[] = (string) $entry->getAdditionalTransactionInformation();
        }
        catch (BadMethodCallException $e) {
          // no handling, cannot test existance of AdditionalTransactionInformation any other way
          // then this, which might throw BadMethodCallException
        }

        // VDK
        try {
          $information[] = (string) $entry->getRemittanceInformation()->getMessage();
        }
        catch (BadMethodCallException $e) {
          // no handling, cannot test existance of AdditionalTransactionInformation any other way
          // then this, which might throw BadMethodCallException
        }
        catch (\Throwable $t) {
          // no handling
        }

        $transaction_data['information'] = implode(
          ' ',
          array_filter(
            array_map(
              function ($s) {
                return trim(preg_replace('/\s+/', ' ', $s));
              },
              $information
            )
          )
        );

        foreach ($entry->getRelatedParties() as $party) {
          if (($amount->getAmount() < 0 && ($party->getRelatedPartyType() instanceof Creditor)) ||
            ($amount->getAmount() > 0 && ($party->getRelatedPartyType() instanceof Debtor))) {
            $transaction_data['account_holder_name'] = $party->getRelatedPartyType()->getName();
            $transaction_data['account_holder_address'] = CAMT053::formatAddress($party->getRelatedPartyType()->getAddress());
            try {
              $transaction_data['account_number'] = $party->getAccount()->getIdentification();
            }
            catch (BadMethodCallException $e) {
              // no handling, cannot test existance of Account for related party any other way
              // then this, which might throw BadMethodCallException
            }
          }
        }

        \Drupal::service('banking_import.default')->createImportedTransaction($transaction_data);
        $n++;
      }

      if (!isset($context['results']['count'])) {
        $context['results']['count'] = 0;
      }

      $context['results']['count'] += 1;
    }
  }

}
