<?php

namespace Drupal\banking_ogm\Entity;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\user\UserInterface;

/**
 * Defines the Banking OGM entity.
 *
 * @ingroup banking_ogm
 *
 * @ContentEntityType(
 *   id = "banking_ogm",
 *   label = @Translation("Banking OGM"),
 *   bundle_label = @Translation("Banking OGM type"),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\banking_ogm\BankingOGMListBuilder",
 *     "views_data" = "Drupal\banking_ogm\Entity\BankingOGMViewsData",
 *
 *     "form" = {
 *       "default" = "Drupal\banking_ogm\Form\BankingOGMForm",
 *       "add" = "Drupal\banking_ogm\Form\BankingOGMForm",
 *       "edit" = "Drupal\banking_ogm\Form\BankingOGMForm",
 *       "delete" = "Drupal\banking_ogm\Form\BankingOGMDeleteForm",
 *     },
 *     "access" = "Drupal\banking_ogm\BankingOGMAccessControlHandler",
 *     "route_provider" = {
 *       "html" = "Drupal\banking_ogm\BankingOGMHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "banking_ogm",
 *   admin_permission = "administer banking ogm entities",
 *   entity_keys = {
 *     "id" = "id",
 *     "bundle" = "type",
 *     "label" = "name",
 *     "uuid" = "uuid",
 *     "uid" = "user_id",
 *     "langcode" = "langcode",
 *     "status" = "status",
 *   },
 *   links = {
 *     "canonical" = "/admin/banking/banking_ogm/{banking_ogm}",
 *     "add-page" = "/admin/banking/banking_ogm/add",
 *     "add-form" = "/admin/banking/banking_ogm/add/{banking_ogm_type}",
 *     "edit-form" = "/admin/banking/banking_ogm/{banking_ogm}/edit",
 *     "delete-form" = "/admin/banking/banking_ogm/{banking_ogm}/delete",
 *     "collection" = "/admin/banking/banking_ogm",
 *   },
 *   bundle_entity_type = "banking_ogm_type",
 *   field_ui_base_route = "entity.banking_ogm_type.edit_form"
 * )
 */
class BankingOGM extends ContentEntityBase implements BankingOGMInterface {

  use EntityChangedTrait;

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage_controller, array &$values) {
    parent::preCreate($storage_controller, $values);
    $values += [
      'user_id' => \Drupal::currentUser()->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage) {
    parent::preSave($storage);

    if (!$this->getName()) {
      $connection = \Drupal::database();
      $transaction = $connection->startTransaction();

      $bundle_config = \Drupal::entityTypeManager()
              ->getStorage('banking_ogm_type')
              ->load($this->bundle());
      $prefix = $bundle_config->getPrefix();

      if (! $prefix) {
        $prefix = '000';
      }

      $result = $connection->query(
        "select max(convert(coalesce(name, 0), SIGNED INTEGER)) + 1 as maxid from banking_ogm where name like :prefix",
        [
          ':prefix' => "{$prefix}%",
        ]
      );
      $row = $result->fetchObject();

      if (! $row->maxid) {
        $next = $prefix.str_pad($row->maxid, 10 - strlen($prefix), '0', STR_PAD_LEFT);
      } else {
        $next = str_pad($row->maxid, 10, '0', STR_PAD_LEFT);
      }

      $this->setName($next);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getName() {
    return $this->get('name')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setName($name) {
    $this->set('name', $name);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormattedOGM() {
    $number = $this->get('name')->value;

    $msg = str_pad(substr($number, 0, 10), 10, "0", STR_PAD_LEFT);
    $rest = $msg % 97;

    if ($rest === 0) {
      $rest = 97;
    }
    if ($rest < 10) {
      $rest = "0" . $rest;
    }

    $ogm = $msg.$rest;
    return "+++".substr($ogm,0,3)."/".substr($ogm,3,4)."/".substr($ogm,7,5)."+++";
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime($timestamp) {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwner() {
    return $this->get('user_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwnerId() {
    return $this->get('user_id')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwnerId($uid) {
    $this->set('user_id', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwner(UserInterface $account) {
    $this->set('user_id', $account->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function isPublished() {
    return (bool) $this->getEntityKey('status');
  }

  /**
   * {@inheritdoc}
   */
  public function setPublished($published) {
    $this->set('status', $published ? TRUE : FALSE);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['user_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Authored by'))
      ->setDescription(t('The user ID of author of the Banking OGM entity.'))
      ->setRevisionable(TRUE)
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setTranslatable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'author',
        'weight' => 0,
      ])
      // ->setDisplayOptions('form', [
      //   'type' => 'entity_reference_autocomplete',
      //   'weight' => 5,
      //   'settings' => [
      //     'match_operator' => 'CONTAINS',
      //     'size' => '60',
      //     'autocomplete_type' => 'tags',
      //     'placeholder' => '',
      //   ],
      // ])
      // ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Name'))
      ->addConstraint('UniqueField')
      ->setDescription(t('The name of the Banking OGM entity.'))
      ->setSettings([
        'max_length' => 50,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Publishing status'))
      ->setDescription(t('A boolean indicating whether the Banking OGM is published.'))
      ->setDefaultValue(TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the entity was last edited.'));

    return $fields;
  }

}
