<?php
namespace Drupal\banking_webform_payment\Plugin\WebformHandler;

use Drupal\Core\Form\FormStateInterface;

use Drupal\webform\webformSubmissionInterface;

use Drupal\accounting_webform\Plugin\WebformHandler\AccountingAccountWebformHandler;

use Drupal\banking_ogm\Entity\BankingOGM;

/**
 * Form submission handler.
 *
 * @WebformHandler(
 *   id = "banking_webform_payment_handler",
 *   label = @Translation("Banking Webform Payment Handler"),
 *   category = @Translation("Webform Handler"),
 *   description = @Translation("Create OGM to follow up payment"),
 *   cardinality =
 *       \Drupal\webform\Plugin\WebformHandlerInterface::CARDINALITY_SINGLE,
 *   results =
 *    \Drupal\webform\Plugin\WebformHandlerInterface::RESULTS_PROCESSED,
 * )
 */
class BankingWebformPaymentHandler extends AccountingAccountWebformHandler {

  /**
   * {@inheritdoc}
   */
  public function getOGM($submission) {
    $account = $this->getAccount($submission);
    if (!$account) {
      return;
    }

    $ogm_ids = \Drupal::EntityQuery('banking_ogm')
             ->condition('type', $this->configuration['ogm_bundle'], '=')
             ->condition('account', $account->id(), '=')
             ->execute();
    if (count($ogm_ids)) {
      $ogm = BankingOGM::load(reset($ogm_ids));
    } else {
      $ogm = BankingOGM::create([
               'type' => $this->configuration['ogm_bundle'],
               'account' => $account->id(),
      ]);
      $ogm->save();
    }
    return $ogm->getFormattedOGM();
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(WebformSubmissionInterface $webform_submission, $update = TRUE) {
    $this->getOGM($webform_submission);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state, WebformSubmissionInterface $webform_submission) {
    parent::submitForm($form, $form_state, $webform_submission);
  }

  /**
   * {@inheritdoc}
   */
  public function confirmForm(array &$form, FormStateInterface $form_state, WebformSubmissionInterface $webform_submission) {
    parent::confirmForm($form, $form_state, $webform_submission);
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return parent::defaultConfiguration() + [
      'ogm_bundle' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['additional']['ogm_bundle'] = [
      '#title' => 'OGM Bundle',
      '#type' => 'textfield',
      '#parents' => ['settings', 'ogm_bundle'],
      '#default_value' => $this->configuration['ogm_bundle'],
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::validateConfigurationForm($form, $form_state);

    $ogm_bundle = \Drupal::entityTypeManager()
                ->getStorage('banking_ogm_type')
                ->load($form_state->getValue('ogm_bundle'));
    if (!$ogm_bundle) {
      $form_state->setErrorByName(
        'ogm_bundle',
        $this->t('Invalid OGM Bundle @bundle', [
            '@bundle' => $form_state->getValue('currency'),
        ]));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    parent::applyFormStateToConfiguration($form_state);
  }

}
