<?php

namespace Drupal\banking\Entity;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\user\UserInterface;

use Drupal\banking\Exception\BankingIntegrityException;

/**
 * Defines the Bank transaction entity.
 *
 * @ingroup banking
 *
 * @ContentEntityType(
 *   id = "bank_transaction",
 *   label = @Translation("Bank transaction"),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\banking\BankTransactionListBuilder",
 *     "views_data" = "Drupal\banking\Entity\BankTransactionViewsData",
 *     "translation" = "Drupal\banking\BankTransactionTranslationHandler",
 *
 *     "form" = {
 *       "default" = "Drupal\banking\Form\BankTransactionForm",
 *       "add" = "Drupal\banking\Form\BankTransactionForm",
 *       "edit" = "Drupal\banking\Form\BankTransactionForm",
 *       "delete" = "Drupal\banking\Form\BankTransactionDeleteForm",
 *     },
 *     "access" = "Drupal\banking\BankTransactionAccessControlHandler",
 *     "route_provider" = {
 *       "html" = "Drupal\banking\BankTransactionHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "bank_transaction",
 *   admin_permission = "administer bank transaction entities",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "name",
 *     "uuid" = "uuid",
 *     "uid" = "user_id",
 *     "langcode" = "langcode",
 *     "status" = "status",
 *   },
 *   links = {
 *     "canonical" = "/admin/banking/bank_transaction/{bank_transaction}",
 *     "add-form" = "/admin/banking/bank_transaction/add",
 *     "edit-form" = "/admin/banking/bank_transaction/{bank_transaction}/edit",
 *     "delete-form" = "/admin/banking/bank_transaction/{bank_transaction}/delete",
 *     "state-form" = "/admin/banking/bank_transaction/{bank_transaction}/state",
 *     "collection" = "/admin/banking/bank_transaction",
 *   },
 *   field_ui_base_route = "bank_transaction.settings"
 * )
 */
class BankTransaction extends ContentEntityBase implements BankTransactionInterface {

  use EntityChangedTrait;

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage_controller, array &$values) {
    parent::preCreate($storage_controller, $values);
    $values += [
      'user_id' => \Drupal::currentUser()->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function delete() {
    if ($this->getState() != 'draft') {
      throw new BankingIntegrityException(
        "You cannot delete a non-draft Bank Transaction");
    }
    return parent::delete();
  }

  /**
   * {@inheritdoc}
   */
  public function getName() {
    return $this->get('name')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setName($name) {
    $this->set('name', $name);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setState($state) {
    $this->set('state', $state);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getAmount() {
    return $this->get('amount')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getDate() {
    return $this->get('date')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getState() {
    return $this->get('state')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getInformation() {
    return $this->get('information')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getAccountNumber() {
    return $this->get('account_number')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getAccountHolderName() {
    return $this->get('account_holder_name')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getAccountHolderAddress() {
    return $this->get('account_holder_address')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getBankAccount() {
    return $this->get('bank_account_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime($timestamp) {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwner() {
    return $this->get('user_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwnerId() {
    return $this->get('user_id')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwnerId($uid) {
    $this->set('user_id', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwner(UserInterface $account) {
    $this->set('user_id', $account->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function isPublished() {
    return (bool) $this->getEntityKey('status');
  }

  /**
   * {@inheritdoc}
   */
  public function setPublished($published) {
    $this->set('status', $published ? TRUE : FALSE);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getData($key, $default = NULL) {
    $data = [];
    if (!$this->get('extra')->isEmpty()) {
      $data = $this->get('extra')->first()->getValue();
    }
    return isset($data[$key]) ? $data[$key] : $default;
  }

  /**
   * {@inheritdoc}
   */
  public function setData($key, $value) {
    $this->get('extra')->__set($key, $value);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['user_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Authored by'))
      ->setDescription(t('The user ID of author of the Bank transaction entity.'))
      ->setRevisionable(TRUE)
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setTranslatable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'author',
        'weight' => 0,
      ])
      // ->setDisplayOptions('form', [
      //   'type' => 'entity_reference_autocomplete',
      //   'weight' => 5,
      //   'settings' => [
      //     'match_operator' => 'CONTAINS',
      //     'size' => '60',
      //     'autocomplete_type' => 'tags',
      //     'placeholder' => '',
      //   ],
      // ])
      // ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['bank_account_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Bank Account'))
      ->setDescription(t('The Bank Account.'))
      ->setSettings(array(
        'target_type' => 'bank_account',
        'default_value' => 0,
      ))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'author',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 1,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Name'))
      ->setDescription(t('The name / ID of the Bank transaction entity.'))
      ->setSettings([
        'max_length' => 50,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['date'] = BaseFieldDefinition::create('datetime')
      ->setLabel(t('Transaction Date'))
      ->setDescription(t('The Date of the transaction.'))
      ->setSettings(array(
          'datetime_type' => 'date'
      ))
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['amount'] = BaseFieldDefinition::create('decimal')
      ->setLabel(t('Amount'))
      ->setDescription(t('The Amount of the transactioin.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'number_decimal',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'number',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['account_holder_name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Account Holder Name'))
      ->setDescription(t('The Name of the Account Holder.'))
      ->setSettings([
        'max_length' => 50,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['account_holder_address'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Account Holder Address'))
      ->setDescription(t('The Address of the Account Holder.'))
      ->setSettings([
        'max_length' => 250,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);


    $fields['account_number'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Account Number'))
      ->setDescription(t('The Account Number of the other Party.'))
      ->setSettings([
        'max_length' => 50,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['iban'] = BaseFieldDefinition::create('iban_bic')
      ->setLabel(t('IBAN'))
      ->setDescription(t('The Account Number of the other Party.'))
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'iban_bic_widget',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);


    $fields['information'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Information'))
      ->setDescription(t('The information field of the Bank transaction.'))
      ->setDefaultValue('')
      ->setSetting('max_length', 4096)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['extra'] = BaseFieldDefinition::create('map')
      ->setLabel(t('Extra'))
      ->setDescription(t('Extra data for this Bank Transaction.'));

    $fields['state'] = BaseFieldDefinition::create('simple_workflow_state_field')
      ->setLabel(t('State'))
      ->setDescription(t('State of the Transaction.'))
      ->setRevisionable(TRUE)
      ->setDefaultValue('draft');

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Publishing status'))
      ->setDescription(t('A boolean indicating whether the Bank transaction is published.'))
      ->setDefaultValue(TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the entity was last edited.'));

    return $fields;
  }

}
