<?php

namespace Drupal\dsr_dashboard\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Url;

use Drupal\dsr\DsrDrushServiceInterface;
use Drupal\shareholder_register\Entity\ShareType;

/**
 * Provides a 'Dsr Infographic' Block.
 *
 * @Block(
 *   id = "dsr_infographic_block",
 *   admin_label = @Translation("DSR Infographic Block"),
 *   category = @Translation("DSR Infographic Block"),
 * )
 */
class DsrInfographicBlock extends BlockBase {

  private $palette = [
    "#aadae2",
    "#dd514e",
    "#abecec",
    "#e51000",
    "#42c8c6",
    "#6b0603",
    "#00bee7",
    "#09394c",
  ];

  /**
   * {@inheritdoc}
   */
  private function get_share_types_donut() {
    $serie = [
      "name" => 'shares',
      "color" => $this->palette[0],
      "type" => NULL,
      "data" => [],
    ];
    $categories = [];

    foreach (
      \Drupal::database()->query("
          SELECT count(*) as total,
          share_type_id
          FROM share as s
          WHERE s.state = 'issued'
          GROUP BY s.share_type_id
        ") as $row) {
      $serie["data"][] = intval($row->total);
      $type = ShareType::load($row->share_type_id)->getName();
      if ($type) {
        $categories[] = $type;
      }
    }

    $build = [
      '#theme' => 'charts_blocks',
      '#library' => 'google',
      '#title' => 'Shares by type',
      '#categories' => $categories,
      '#seriesData' => [$serie],
      '#options' => [
        'title' => 'shares by type',
        'library' => 'google',
        'type' => 'donut',
        'colors' => $this->palette,
        'legend_position' => 'none',
        'title_position' => 'none',
        'background' => 'transparent',
      ],
      '#secondaryOptions' => [],
      '#id' => "chart-shares-by-type",
      '#cache' => \Drupal::entityTypeManager()->getDefinition('share_transaction')->getListCacheTags(),
      '#attributes' => [
        'class' => [
          'data-chart-shares-by-type',
        ],
      ],
    ];
    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function build() {

    // Singup forms.
    $signup_forms_by_state = [];
    foreach (\Drupal::database()->query("
          SELECT state, count(*) as count
          FROM accounting_account
          WHERE type='intekenformulier'
          GROUP BY state") as $row) {
      $signup_forms_by_state[$row->state] = $row->count;
    }
    $signup_forms = \Drupal::database()
      ->query("select count(*) as count from accounting_account where type = 'intekenformulier' and state='open'")
      ->fetchObject()->count;

    // Pending sign up account amount.
    $q = \Drupal::database()->query("select coalesce(sum(credit - debit), 0) as sum from journal_entry_line inner join accounting_account
    on (journal_entry_line.accounting_account_id = accounting_account.id) where accounting_account.type = 'intekenformulier'
    and accounting_account.state in ('open', 'pending')");
    $pending_amount = $q->fetchObject()->sum;

    // Draft bank transaction amount.
    $q = \Drupal::database()->query("select coalesce(sum(abs(amount)), 0) as sum from bank_transaction where state='draft'");
    $draft_amount = $q->fetchObject()->sum;
    $q = \Drupal::database()->query("select max(date) as last_import from bank_transaction");
    $last_import = $q->fetchObject()->last_import;
    $module_path = \Drupal::service('module_handler')->getModule('dsr_dashboard')->getPath();

    return [
      '#markup' => "<img class=\"infographic\" src=\"/{$module_path}/images/dashboard.svg\"/>",
      '#prefix' => '<div class="dsr-dashboard-infographic">',
      '#suffix' => '</div>',
      [
        [
          '#type' => 'link',
          '#title' => $signup_forms,
          '#url' => Url::fromRoute('view.signup_forms.page_1'),
          '#attributes' => [
            'class' => [
              'data-signup-forms',
            ],
          ],
          '#id' => 'data-signup-forms',
        ],
        [
          '#type' => 'link',
          '#title' => \Drupal::entityQuery('share_transaction')->accessCheck(FALSE)->condition('state', 'draft')->count()->execute(),
          '#url' => Url::fromRoute('view.share_transactions.page_2'),
          '#attributes' => [
            'class' => [
              'data-pending-transactions',
            ],
          ],
          '#id' => 'data-pending-transactions',
        ],
        [
          '#type' => 'link',
          '#title' => $draft_amount,
          '#url' => Url::fromRoute('entity.bank_transaction.collection'),
          '#attributes' => [
            'class' => [
              'data-draft-bank-amount',
            ],
          ],
          '#id' => 'data-draft-bank-amount',
        ],
        [
          '#type' => 'link',
          '#title' => $last_import,
          '#url' => Url::fromRoute('banking_import.camt_import_form'),
          '#attributes' => [
            'class' => [
              'data-last-import',
            ],
          ],
          '#id' => 'data-last-import',
        ],
        [
          '#type' => 'link',
          '#title' => $pending_amount,
          '#url' => Url::fromRoute('view.signup_forms.page_3'),
          '#attributes' => [
            'class' => [
              'data-pending-amount',
            ],
          ],
          '#id' => 'data-pending-amount',
        ],
        [
          '#type' => 'link',
          '#title' => \Drupal::entityQuery('shareholder')->accessCheck(FALSE)->condition('state', 'valid')->condition('current', '1')->count()->execute(),
          '#url' => Url::fromRoute('entity.shareholder.collection'),
          '#attributes' => [
            'class' => [
              'data-shareholders',
            ],
          ],
          '#id' => 'data-shareholders',
        ],
        // $build,
        $this->get_share_types_donut(),
      ],
      '#attached' => [
        'library' => [
          'dsr_dashboard/global',
        ],
      ],
    ];
  }

}
