<?php

namespace Drupal\dsr_deposito\EventSubscriber;

use Symfony\Component\EventDispatcher\EventSubscriberInterface;

use Drupal\accounting\Event\JournalEntryLineEvent;
use Drupal\accounting\Entity\AccountingAccount;

use Drupal\banking_ogm\Entity\BankingOGM;

use Drupal\shareholder_register\ShareholderRegisterServiceInterface;
use Drupal\shareholder_register\Entity\ShareIssue;
use Drupal\shareholder_register\Entity\ShareTransaction;
use Drupal\shareholder_register\Entity\ShareTransactionGroupType;

use Drupal\shareholder_register\Event\ShareholderEvent;

/**
 * Class DSRDepositoEventSubscriber.
 *
 * @package Drupal\dsr_deposito
 */
class DSRDepositoEventSubscriber implements EventSubscriberInterface {

  /**
   * Drupal\shareholder_register\ShareholderRegisterServiceInterface definition.
   *
   * @var \Drupal\shareholder_register\ShareholderRegisterServiceInterface
   */
  protected $shareholderRegisterService;

  /**
   * Constructs a new DsrSignUpService object.
   */
  public function __construct(ShareholderRegisterServiceInterface $shareholderRegisterService) {
    $this->shareholderRegisterService = $shareholderRegisterService;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    $events[ShareholderEvent::VALIDATED_EVENT_NAME][] = [
      'createShareholderDepositoAccount',
      800,
    ];
    $events['rules_journal_entry_line_validated'][] = [
      'createShareTransactionForDepositoAccount',
      800,
    ];
    return $events;
  }

  /**
   * Subscriber for the Shareholder validated event.
   *
   * @param \Drupal\shareholder_register\Event\ShareholderEvent $event
   *   The event.
   */
  public function createShareholderDepositoAccount(ShareholderEvent $event) {
    if ($event->getShareholder()) {
      $shareholder = $event->getShareholder();

      $ogm_ids = \Drupal::entityQuery('banking_ogm')
        ->accessCheck(FALSE)
        ->condition('type', 'shareholder')
        ->condition('account.entity.type', 'deposito')
        ->condition('account.entity.field_shareholder.target_id', $shareholder->id())
        ->execute();

      if (!count($ogm_ids)) {
        $account = AccountingAccount::create([
          'type' => 'deposito',
          'field_shareholder' => $shareholder->id(),
          'name' => t('shareholder account: @number @name', [
            '@number' => $shareholder->getNumber(),
            '@name' => $shareholder->getName(),
          ]),
        ]);
        $account->save();

        $ogm = BankingOGM::create([
          'type' => 'shareholder',
          'account' => $account->id(),
        ]);
        $ogm->save();
        \Drupal::logger('dsr_deposito')->notice('created deposito account for shareholder ');
      }
    }
  }

  /**
   * Subscriber Callback for the event.
   *
   * @param \Drupal\accounting\Event\JournalEntryLineEvent $event
   *   The event.
   */
  public function createShareTransactionForDepositoAccount(JournalEntryLineEvent $event) {
    $line = $event->journal_entry_line;
    $account = $line->getAccount();

    $config = \Drupal::config('dsr_deposito.settings');

    if ($account->bundle() != $config->get('deposito_account_bundle')) {
      return;
    }
    $deposito_field_definitions = \Drupal::service('entity_field.manager')->getFieldDefinitions(
      'accounting_account', $config->get(
        'deposito_account_bundle'));

    $balance = min(
      $account->getCreditBalance(),
      $account->getBalance($line->getPeriodDate())['balance'] * -1
    );

    if (isset($deposito_field_definitions['field_share_issue']) && $account->get('field_share_issue')->entity) {
      $share_issue = $account->get('field_share_issue')->entity;
    }
    else {
      $share_issue = ShareIssue::load($config->get('share_issue'));
    }

    if (!$share_issue || $balance < $share_issue->getSharePrice()) {
      \Drupal::logger('dsr_deposito')->notice('deposito account @account has insuffucient balance: @balance',
        [
          '@account' => $account->getName(),
          '@balance' => $account->getCreditBalance(),
        ]
      );
      return;
    }

    if ($account->getState() != 'open') {
      \Drupal::logger('dsr_deposito')->notice(
        'deposito account: account not in state open: @account',
        ['@account' => $account->getName()]);
      return;
    }

    $transaction_ids = \Drupal::entityQuery('share_transaction')
      ->accessCheck(FALSE)
      ->condition('field_account', $account->id())
      ->condition('state', 'draft')
      ->execute();

    $used_balance = 0;
    foreach (ShareTransaction::loadMultiple($transaction_ids) as $t) {
      if ($t->getShareGroupTypeBaseType() != ShareTransactionGroupType::BASE_ISSUE) {
        continue;
      }
      $used_balance = bcadd(
        $used_balance,
        bcmul(
          $t->getQuantity(),
          $t->getShareIssue()->getSharePrice(),
          2
        ),
        2
      );
    }

    $free_balance = bcsub(
      $balance,
      $used_balance,
      2
    );

    $share_count = floor(
      $free_balance
      / $share_issue->getSharePrice()
    );

    if ($share_count > 0) {
      if (!$account->hasField('field_shareholder')) {
        \Drupal::logger('dsr_deposito')->notice(
          "deposito account: account @account doesn't have a field_shareholder",
          ['@account' => $account->getName()]);
        return;
      }

      $shareholder = $account->get('field_shareholder')->entity;

      if (!$shareholder) {
        // Attempt to create a shareholder, if there is an attached submission.
        if ($account->hasField('field_submission') &&
          $submission = $account->get('field_submission')->entity) {

          $handler_collection = $submission->getWebform()->getHandlers('shareholder_mapping_webform_handler');
          $instance_ids = $handler_collection->getInstanceIds();
          if (!count($instance_ids)) {
            \Drupal::logger('dsr_deposito')->notice(
              "deposito account: account @account shareholder unset and no shareholder_mapping_webform_handler",
              ['@account' => $account->getName()]);
            return;
          }
          $handler = $handler_collection->get(reset($instance_ids));

          $shareholder = $handler->createOrGetShareholder($submission);
          $account->set('field_shareholder', $shareholder)->save();
        }
        else {
          \Drupal::logger('dsr_deposito')->notice(
            "deposito account: account @account shareholder unset and no submission",
            ['@account' => $account->getName()]);
          return;
        }
      }

      $transaction = $this->shareholderRegisterService->createIssueTransaction(
        $shareholder,
        $share_issue,
        $share_count,
        $account,
        $line->getPeriodDate()
      );

      // Copy field_share_x fields to transaction if they exist.
      foreach ($deposito_field_definitions as $k => $v) {
        if (substr($k, 0, strlen('field_share_')) === 'field_share_' &&
          $transaction->hasField('field_' . substr($k, strlen('field_share_')))) {
          $transaction->set(
            'field_' . substr($k, strlen('field_share_')),
            $account->get($k)->getValue()
          );
          $transaction->save();
        }
      }
    }
  }

}
