<?php

namespace Drupal\Tests\dsr_deposito\Kernel;

use Drupal\field\Entity\FieldStorageConfig;
use Drupal\field\Entity\FieldConfig;

use Drupal\shareholder_register\Entity\ShareTransaction;
use Drupal\shareholder_register\Entity\ShareType;
use Drupal\shareholder_register\Entity\ShareIssue;

use Drupal\accounting\Entity\AccountingAccount;
use Drupal\accounting\Entity\AccountingJournal;

use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\shareholder_register\Traits\LoadRegisterTrait;

/**
 * Tests deposito transactions.
 *
 * @group shareholder_register
 */
class DepositoTest extends KernelTestBase {

  use LoadRegisterTrait;

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'system',
    'user',
    'file',
    'text',
    'field',
    'datetime',
    'views',
    'views_bulk_operations',
    'vbo_export',
    'workflows',
    'simple_workflows',
    'iban_bic_field',
    'accounting',
    'banking',
    'banking_ogm',
    'banking_accounting',
    'shareholder_register',
    'shareholder_register_accounting',
    'dsr',
    'dsr_deposito',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->traitSetUp();
    $this->installEntitySchema('accounting_account');
    $this->installEntitySchema('accounting_journal');
    $this->installEntitySchema('journal_entry');
    $this->installEntitySchema('journal_entry_line');
    $this->installEntitySchema('reconciliation');

    $this->installEntitySchema('banking_ogm');

    $this->installConfig('accounting');
    $this->installConfig('shareholder_register_accounting');
    $this->installConfig('dsr_deposito');

    $a1 = AccountingAccount::create([
      'type' => 'kapitaal',
    ]);
    $a1->save();
    $a2 = AccountingAccount::create([
      'type' => 'suspense_account',
    ]);
    $a2->save();

    $j1 = AccountingJournal::create([
      'name' => 'Diverse',
    ]);
    $j1->save();

    \Drupal::service('config.factory')->getEditable('shareholder_register_accounting.settings')
      ->set('share_transaction_account',
        $a2->id())
      ->set('share_transaction_journal',
        $a1->id())
      ->set('share_value_account',
        $j1->id())
      ->save();

    \Drupal::service('config.factory')->getEditable('dsr_deposito.settings')
      ->set('deposito_account_bundle', 'deposito')
      ->save();

    $this->deposit_account = AccountingAccount::create([
      'type' => 'suspense_account',
    ]);
    $this->deposit_account->save();

    $this->deposit_journal = AccountingJournal::create([
      'name' => 'Manual',
    ]);
    $this->deposit_journal->save();

  }

  /**
   * Setup test register for negative transactions.
   */
  protected function loadTestRegister() {
    $register = $this->loadBasicRegister(
      [
        'shareholders' => [
          [
            'number' => 1,
          ],
        ],
        'transactions' => [
          [
            'quantity' => 5,
            'shareholder_number' => 1,
            'payment_date' => '2017-01-02',
            'validation_date' => '2017-02-02',
          ],
        ],
      ]);

    \Drupal::service('config.factory')->getEditable('dsr_deposito.settings')
      ->set('share_issue', $register['issue'])
      ->save();

    $shareholder_id = reset($register['shareholder']);
    $account_ids = \Drupal::entityQuery('accounting_account')
      ->accessCheck(FALSE)
      ->condition('type', 'deposito')
      ->condition('field_shareholder', $shareholder_id)
      ->execute();

    if (count($account_ids) !== 1) {
      throw new \Exception('internal test error');
    }
    $register['deposito_account'] = reset($account_ids);

    return $register;
  }

  /**
   * Test a simple deposito transaction.
   */
  public function testDepositoTransaction() {
    $register = $this->loadTestRegister();

    \Drupal::service('accounting.book_entry')
      ->bookEntry(
        'default',
        '2020-01-01',
        $this->deposit_journal,
        [
          [
            'name' => 'manual test deposit',
            'amount' => bcmul('-1', 200, 2),
            'account_id' => $register['deposito_account'],
          ],
          [
            'name' => 'manual test deposit',
            'amount' => 200,
            'account_id' => $this->deposit_account->id(),
          ],
        ]
      );

    $transaction_ids = \Drupal::entityQuery('share_transaction')
      ->accessCheck(FALSE)
      ->condition('shareholder_id', reset($register['shareholder']))
      ->condition('state', 'draft')
      ->execute();

    $this->assertEquals(count($transaction_ids), 1);

    $t = ShareTransaction::load(reset($transaction_ids));

    $this->assertEquals($t->getPaymentDate(), '2020-01-01');
    $this->assertEquals($t->getQuantity(), 2);
  }

  /**
   * Test a deposito transaction paid in 2 parts.
   */
  public function testDepositoTransactionPaidInTwoParts() {
    $register = $this->loadTestRegister();

    \Drupal::service('accounting.book_entry')
      ->bookEntry(
        'default',
        '2020-01-01',
        $this->deposit_journal,
        [
          [
            'name' => 'manual test deposit',
            'amount' => bcmul('-1', 50, 2),
            'account_id' => $register['deposito_account'],
          ],
          [
            'name' => 'manual test deposit',
            'amount' => 50,
            'account_id' => $this->deposit_account->id(),
          ],
        ]
      );
    \Drupal::service('accounting.book_entry')
      ->bookEntry(
        'default',
        '2020-02-01',
        $this->deposit_journal,
        [
          [
            'name' => 'manual test deposit',
            'amount' => bcmul('-1', 100, 2),
            'account_id' => $register['deposito_account'],
          ],
          [
            'name' => 'manual test deposit',
            'amount' => 100,
            'account_id' => $this->deposit_account->id(),
          ],
        ]
      );

    $transaction_ids = \Drupal::entityQuery('share_transaction')
      ->accessCheck(FALSE)
      ->condition('shareholder_id', reset($register['shareholder']))
      ->condition('state', 'draft')
      ->execute();

    $this->assertEquals(count($transaction_ids), 1);

    $t = ShareTransaction::load(reset($transaction_ids));

    $this->assertEquals($t->getPaymentDate(), '2020-02-01');
    $this->assertEquals($t->getQuantity(), 1);
  }

  /**
   * Test a deposito transaction with attributes.
   */
  public function testDepositoTransactionWithAttribute() {
    $register = $this->loadTestRegister();

    // Create field on deposito account and on share_transaction.
    FieldStorageConfig::create(
      [
        'field_name' => 'field_share_test',
        'entity_type' => 'accounting_account',
        'type' => 'text',
        'cardinality' => 1,
      ]
    )->save();
    FieldConfig::create(
      [
        'field_name' => 'field_share_test',
        'entity_type' => 'accounting_account',
        'bundle' => 'deposito',
        'label' => 'A Test field',
      ]
    )->save();

    FieldStorageConfig::create(
      [
        'field_name' => 'field_test',
        'entity_type' => 'share_transaction',
        'type' => 'text',
        'cardinality' => 1,
      ]
    )->save();
    FieldConfig::create(
      [
        'field_name' => 'field_test',
        'entity_type' => 'share_transaction',
        'bundle' => 'share_transaction',
        'label' => 'A Test field',
      ]
    )->save();

    // Set the value on the deposito account.
    $a = AccountingAccount::load($register['deposito_account']);
    $a->set('field_share_test', 'kernel_test')->save();

    \Drupal::service('accounting.book_entry')
      ->bookEntry(
        'default',
        '2020-01-01',
        $this->deposit_journal,
        [
          [
            'name' => 'manual test deposit',
            'amount' => bcmul('-1', 200, 2),
            'account_id' => $register['deposito_account'],
          ],
          [
            'name' => 'manual test deposit',
            'amount' => 200,
            'account_id' => $this->deposit_account->id(),
          ],
        ]
      );

    $transaction_ids = \Drupal::entityQuery('share_transaction')
      ->accessCheck(FALSE)
      ->condition('shareholder_id', reset($register['shareholder']))
      ->condition('state', 'draft')
      ->execute();

    $this->assertEquals(count($transaction_ids), 1);

    $t = ShareTransaction::load(reset($transaction_ids));

    $this->assertEquals($t->getPaymentDate(), '2020-01-01');
    $this->assertEquals($t->getQuantity(), 2);
    $this->assertEquals($t->get('field_test')->value, 'kernel_test');
  }

  /**
   * Test a deposito transaction with share type attribute.
   */
  public function testDepositoTransactionWithAttributeShareType() {
    $register = $this->loadTestRegister();

    $type = ShareType::create(
      [
        'name' => 'aandeel B',
        'par_value' => 100,
      ]
    );
    $type->save();
    $issue = ShareIssue::create(
      [
        'share_type_id' => $type->id(),
      ]
    );
    $issue->save();

    // Create field on deposito account and on share_transaction.
    FieldStorageConfig::create(
      [
        'field_name' => 'field_share_issue',
        'entity_type' => 'accounting_account',
        'type' => 'entity_reference',
        'cardinality' => 1,
        'settings' => [
          'target_type' => 'share_issue',
        ],
      ]
    )->save();
    FieldConfig::create(
      [
        'field_name' => 'field_share_issue',
        'entity_type' => 'accounting_account',
        'bundle' => 'deposito',
        'label' => 'A Test field',
      ]
    )->save();

    // Set the value on the deposito account.
    $a = AccountingAccount::load($register['deposito_account']);
    $a->set('field_share_issue', $issue)->save();

    \Drupal::service('accounting.book_entry')
      ->bookEntry(
        'default',
        '2020-01-01',
        $this->deposit_journal,
        [
          [
            'name' => 'manual test deposit',
            'amount' => bcmul('-1', 200, 2),
            'account_id' => $register['deposito_account'],
          ],
          [
            'name' => 'manual test deposit',
            'amount' => 200,
            'account_id' => $this->deposit_account->id(),
          ],
        ]
      );

    $transaction_ids = \Drupal::entityQuery('share_transaction')
      ->accessCheck(FALSE)
      ->condition('shareholder_id', reset($register['shareholder']))
      ->condition('state', 'draft')
      ->execute();

    $this->assertEquals(count($transaction_ids), 1);

    $t = ShareTransaction::load(reset($transaction_ids));

    $this->assertEquals($t->getPaymentDate(), '2020-01-01');
    $this->assertEquals($t->getQuantity(), 2);
    $this->assertEquals($t->get('share_issue_id')->target_id, $issue->id());
    $this->assertEquals($t->get('share_type_id')->target_id, $type->id());
  }

}
