<?php

namespace Drupal\dsr_gift_certificates\Plugin\WebformHandler;

use Drupal\Core\Form\FormStateInterface;
use Drupal\webform\Plugin\WebformHandlerBase;
use Drupal\webform\webformSubmissionInterface;

use Drupal\shareholder_register_webform\Element\ShareholderShares;

/**
 * Form submission handler.
 *
 * @WebformHandler(
 *   id = "dsr_gift_certificate_webform_handler",
 *   label = @Translation("DSR Gift Certificate Webform Handler"),
 *   category = @Translation("Webform Handler"),
 *   description = @Translation("DSR Gift Certificate Webform Handler"),
 *   cardinality =
 *       \Drupal\webform\Plugin\WebformHandlerInterface::CARDINALITY_SINGLE,
 *   results =
 *    \Drupal\webform\Plugin\WebformHandlerInterface::RESULTS_PROCESSED,
 * )
 */
class DSRGiftCertificateWebformHandler extends WebformHandlerBase {

  /**
   * {@inheritdoc}
   */
  public function confirmForm(array &$form, FormStateInterface $form_state, WebformSubmissionInterface $submission) {
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state, WebformSubmissionInterface $submission) {
  }

  /**
   * {@inheritdoc}
   */
  public function testPaidInFull(WebformSubmissionInterface $submission, $account) {
    $shareholder_shares_key = $this->configuration['shares_key'];
    $data = $submission->getData();

    $share_price = ShareholderShares::getSharePrice(
      $data[$shareholder_shares_key]['share_issue'],
      $data[$shareholder_shares_key]['share_quantity']
    );

    if ($account->getCreditBalance() >= $share_price['total']) {
      // Generate code.
      $unique = FALSE;
      while (!$unique) {
        $length = 12;
        $x = '0123456789';
        $code = substr(str_shuffle(str_repeat($x, ceil($length / strlen($x)))), 1, $length);

        $unique = !count(\Drupal::Entityquery('accounting_account')
          ->accessCheck(FALSE)
          // FIXME.
          ->condition('field_gift_certificate_code', $code, '=')
          ->execute());
      }
      $account->set('field_gift_certificate_code', $code);

      return TRUE;
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function sendCertificate(WebformSubmissionInterface $submission, $account) {
    $params = [
      'context' => [
        'submission' => $submission,
        'gift_certificate_code' => $account->get('field_gift_certificate_code')->value,
      ],
    ];

    $data = $submission->getData();
    $email = $data[$this->configuration['mail_key']];
    // @todo replace with shareholder preferred language.
    $langcode = \Drupal::currentUser()->getPreferredLangcode();
    $mailManager = \Drupal::service('plugin.manager.mail');
    $result = $mailManager->mail('shareholder_register', $this->configuration['certificate_mail_template'], $email, $langcode, $params, NULL, TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'mail_key' => '',
      'shares_key' => '',
      'certificate_mail_template' => '',
      'certificate_template' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['additional'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Gift Certificate Settings'),
    ];
    $form['additional']['shareholder_shares_key'] = [
      '#title' => 'Shareholder shares element',
      '#type' => 'textfield',
      '#parents' => ['settings', 'shares_key'],
      '#default_value' => $this->configuration['shares_key'],
    ];
    $form['additional']['mail_key'] = [
      '#title' => 'Mail element',
      '#type' => 'textfield',
      '#parents' => ['settings', 'mail_key'],
      '#default_value' => $this->configuration['mail_key'],
    ];
    $form['additional']['certificate_mail_template'] = [
      '#title' => 'Certificate Mail Template',
      '#type' => 'textfield',
      '#parents' => ['settings', 'certificate_mail_template'],
      '#default_value' => $this->configuration['certificate_mail_template'],
    ];
    $form['additional']['certificate_template'] = [
      '#title' => 'Certificate Template',
      '#type' => 'textfield',
      '#parents' => ['settings', 'certificate_template'],
      '#default_value' => $this->configuration['certificate_template'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    parent::applyFormStateToConfiguration($form_state);
  }

}
