<?php

namespace Drupal\dsr_gift_certificates\Plugin\WebformHandler;

use Drupal\Core\Form\FormStateInterface;
use Drupal\webform\Plugin\WebformHandlerBase;
use Drupal\webform\webformSubmissionInterface;

use Drupal\shareholder_register\Entity\ShareIssue;

use Drupal\accounting\Entity\AccountingAccount;

/**
 * Form submission handler.
 *
 * @WebformHandler(
 *   id = "dsr_submit_gift_certificate_webform_handler",
 *   label = @Translation("DSR Submit Gift Certificate Webform Handler"),
 *   category = @Translation("Webform Handler"),
 *   description = @Translation("DSR Submit Gift Certificate Webform Handler"),
 *   cardinality =
 *       \Drupal\webform\Plugin\WebformHandlerInterface::CARDINALITY_SINGLE,
 *   results =
 *    \Drupal\webform\Plugin\WebformHandlerInterface::RESULTS_PROCESSED,
 * )
 */
class DSRSubmitGiftCertificateWebformHandler extends WebformHandlerBase {

  /**
   * {@inheritdoc}
   */
  private function getGiftAccount(WebformSubmissionInterface $submission) {
    $aids = \Drupal::entityQuery('accounting_account')
      ->accessCheck(FALSE)
      ->condition('type', 'cadeau_aandeel')
      ->condition('field_gift_certificate_code', $submission->getData()[$this->configuration['gift_certificate_key']], '=')
      ->execute();

    if (!count($aids)) {
      \Drupal::logger('shareholder_register')->error(
        'DSRSubmitGiftCertificateWebformHandler: account not found for gift certificate @gift',
        [
          '@gift' => $submission->getData()[$this->configuration['gift_certificate_key']],
        ]
      );
      return FALSE;
    }
    $account = AccountingAccount::load(reset($aids));

    if ($account->get('state')->value == 'pending' && !$account->get('field_submission')->entity) {
      return $account;
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function confirmForm(array &$form, FormStateInterface $form_state, WebformSubmissionInterface $submission) {
    $account = $this->getGiftAccount($submission);
    if (!$account) {
      \Drupal::logger('shareholder_register')->error(
        'DSRSubmitGiftCertificateWebformHandler: confirming gift certificate @gift, but account already pending.',
        [
          '@gift' => $submission->getData()[$this->configuration['gift_certificate_key']],
        ]
      );
      return;
    }

    $account->set('field_submission', $submission->id());

    $handler_collection = $submission->getWebform()->getHandlers('shareholder_mapping_webform_handler');
    $instance_ids = $handler_collection->getInstanceIds();
    if (!count($instance_ids)) {
      \Drupal::logger('shareholder_register')->error(
        'DSRSubmitGiftCertificateWebformHandler: no shareholder_mapping_webform_handler handler found for webform!'
      );
      return;
    }
    $handler = $handler_collection->get(reset($instance_ids));

    $account->save();
    $transaction = $handler->performMapping($submission, $account);
    if (!$transaction) {
      \Drupal::logger('shareholder_register')->error(
        'DSRSubmitGiftCertificateWebformHandler: confirming gift certificate @gift, but no transaction was created !',
        [
          '@gift' => $submission->getData()[$this->configuration['gift_certificate_key']],
        ]
      );
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state, WebformSubmissionInterface $submission) {
    if ($this->configuration['shareholder_shares_key'] && $this->configuration['gift_certificate_key']) {
      $account = $this->getGiftAccount($submission);
      if ($account) {
        $values = $form_state->getValue($this->configuration['shareholder_shares_key']);
        $share_issue = ShareIssue::load($values['share_issue']);
        # $shares = intdiv($account->getCreditBalance(), $share_issue->getSharePrice());
        # $shares = max([$shares, 0]);
        $shares = $account->get('field_quantity')->value;
        if ($shares != $form_state->getValue([$this->configuration['shareholder_shares_key'], 'share_quantity'])) {
          $shareholder_shares_data = $submission->getElementData(trim($this->configuration['shareholder_shares_key']));
          $shareholder_shares_data['share_quantity'] = $shares;
          $submission->setElementData(trim($this->configuration['shareholder_shares_key']), $shareholder_shares_data);
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'shareholder_shares_key' => '',
      'gift_certificate_key' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['additional'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Gift Certificate Settings'),
    ];
    $form['additional']['gift_certificate_key'] = [
      '#title' => 'Gift certificate element',
      '#type' => 'textfield',
      '#parents' => ['settings', 'gift_certificate_key'],
      '#default_value' => $this->configuration['gift_certificate_key'],
    ];
    $form['additional']['shareholder_shares_key'] = [
      '#title' => 'Shareholder Shares element',
      '#type' => 'textfield',
      '#parents' => ['settings', 'shareholder_shares_key'],
      '#default_value' => $this->configuration['shareholder_shares_key'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    parent::applyFormStateToConfiguration($form_state);
  }

}
