<?php

namespace Drupal\dsr_sign_up_form\Element;

use Drupal\Component\Utility\Html;
use Drupal\webform\Element\WebformCompositeBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Provides a 'shareholder_identifier'.
 *
 * @FormElement("shareholder_identifier")
 *
 * @see \Drupal\webform\Element\WebformCompositeBase
 * @see \Drupal\dsr_sign_up_form\Element\ShareholderIdentifier
 */
class ShareholderIdentifier extends WebformCompositeBase {

  /**
   * {@inheritdoc}
   */
  public function getInfo() {
    return parent::getInfo() + [
      '#theme' => 'shareholder_identifier',
      '#element_validate' => [[get_class($this), 'validateShareholderIdentifier']],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function getCompositeElements(array $element) {
    // Generate an unique ID that can be used by #states.
    $html_id = Html::getUniqueId('shareholder_identifier');

    $confirm_title = t('I confirm that I am not a shareholder.');
    $confirm_description = t('Confirm that you are not (and have never been) a shareholder, because a shareholder with the same personal data was found. Otherwise, identify yourself as existing shareholder. If you proceed you will receive confirmation after your subscription has been verified.');

    if (isset($element['#confirmation_text']) && $element['#confirmation_text']) {
      $confirm_title = $element['#confirmation_text'];
    }
    if (isset($element['#confirmation_description']) && $element['#confirmation_description']) {
      $confirm_description = $element['#confirmation_description'];
    }

    $elements = [
      'identifier_confirm' => [
        '#type' => 'checkbox',
        '#title' => $confirm_title,
        '#description' => $confirm_description,
        '#states' => [
          'visible' => [
            'input[data-webform-shareholder-identifier-id="' . $html_id . '"]' => ['value' => 'no'],
          ],
        ],
      ],
      'identifier' => [
        '#type' => 'value',
      ],
      'identifier_valid' => [
        '#type' => 'hidden',
        '#value' => 'yes',
        '#attributes' => [
          'data-webform-shareholder-identifier-id' => "{$html_id}",
        ],
      ],
    ];
    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public static function validateShareholderIdentifier(&$element, FormStateInterface $form_state, &$complete_form) {
    // @todo refactor to use shareholder mapping handler, without saving shareholder ?
    $new_identifier = \Drupal::moduleHandler()->invokeAll('sign_up_form_identifier', [$form_state, $complete_form]);

    if (isset($new_identifier['identifier']) && $new_identifier['identifier']) {
      $parents = $element['#parents'];
      $valid = \Drupal::service('shareholder_register.default')->validateIdentifier($new_identifier['identifier']);
      $form_state->set(array_merge($parents, ['identifier']), $new_identifier);
      $element['identifier_valid']['#value'] = $valid ? 'yes' : 'no';
      if (!$valid && !$form_state->getValue(array_merge($parents, ['identifier_confirm']))) {
        $form_state->setError($element['identifier_confirm'], t('Please confirm you are a new shareholder'));
      }
    }
  }

}
