<?php

namespace Drupal\dsr_sign_up_form\Plugin\WebformHandler;

use Drupal\Core\Form\FormStateInterface;
use Drupal\webform\Plugin\WebformHandlerBase;
use Drupal\webform\webformSubmissionInterface;

use Drupal\dsr_sign_up_form\Event\SignUpEvent;

use Drupal\shareholder_register\Entity\ShareIssue;
use Drupal\shareholder_register\Entity\Shareholder;
use Drupal\shareholder_register\Exception\ShareholderRegisterException;
use Drupal\shareholder_register_webform\Element\ShareholderShares;

use Drupal\accounting\Entity\AccountingAccount;

/**
 * Form submission handler.
 *
 * @WebformHandler(
 *   id = "shareholder_mapping_webform_handler",
 *   label = @Translation("Shareholder Mapping webform handler"),
 *   category = @Translation("Webform Handler"),
 *   description = @Translation("Shareholder Mapping"),
 *   cardinality =
 *       \Drupal\webform\Plugin\WebformHandlerInterface::CARDINALITY_UNLIMITED,
 *   results =
 *    \Drupal\webform\Plugin\WebformHandlerInterface::RESULTS_PROCESSED,
 * )
 */
class ShareholderMappingWebformHandler extends WebformHandlerBase {

  /**
   * {@inheritdoc}
   */
  public function postSave(WebformSubmissionInterface $webform_submission, $update = TRUE) {
    $state = $webform_submission->getWebform()->getSetting('results_disabled') ? WebformSubmissionInterface::STATE_COMPLETED : $webform_submission->getState();
    if ($state == WebformSubmissionInterface::STATE_COMPLETED) {
      $should_save = FALSE;
      $shareholder_number_key = $this->configuration['shareholder_number_key'];
      $data = $webform_submission->getData();
      if (isset($data[$shareholder_number_key]['shareholder_id']) && $data[$shareholder_number_key]['shareholder_id']) {
        $shareholder = Shareholder::load($data[$shareholder_number_key]['shareholder_id']);
        \Drupal::service('shareholder_register_webform.mapping')->mapEntityToSubmission(
            $this->configuration['shareholder_mapping'],
            $webform_submission,
            $shareholder
        );
        $should_save = TRUE;
      }
      if (!empty($this->configuration['save_shareholder_id_key'])) {
        $shareholder = $this->createOrGetShareholder($webform_submission);
        $webform_submission->setElementData($this->configuration['save_shareholder_id_key'], $shareholder->id());
        $should_save = TRUE;
      }
      if ($should_save) {
        $webform_submission->resave();
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function mapSubmissionToEntity($mapping, WebformSubmissionInterface $submission, $entity) {
    $data = $submission->getData();

    \Drupal::service('shareholder_register_webform.mapping')->mapSubmissionToEntity(
        $mapping,
        $submission,
        $entity
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getIssueAndQuantity(WebformSubmissionInterface $submission) {
    $shareholder_shares_key = $this->configuration['shares_key'];
    return \Drupal::service('dsr_sign_up_form.default')->getIssueAndQuantity($submission, $shareholder_shares_key);
  }
    
  /**
   * {@inheritdoc}
   */
  public function createOrGetShareholder(WebformSubmissionInterface $submission, bool $save = TRUE) {
    $shareholder_number_key = $this->configuration['shareholder_number_key'];
    $shareholder_type_key = $this->configuration['shareholder_type_key'];
    $data = $submission->getData();

    if (isset($data[$shareholder_number_key]['shareholder_id']) && $data[$shareholder_number_key]['shareholder_id']) {
      $shareholder = Shareholder::load($data[$shareholder_number_key]['shareholder_id']);
    }
    else {
      $shareholder = Shareholder::create([
        'type' => $data[$shareholder_type_key],
      ]);

      $this->mapSubmissionToEntity(
        $this->configuration['shareholder_mapping'],
        $submission,
        $shareholder);

      if ($save) {
        $shareholder->save();
      }
    }

    $event = new SignUpEvent($shareholder, $submission);
    $event_dispatcher = \Drupal::service('event_dispatcher');
    $event_dispatcher->dispatch($event, SignUpEvent::MAPPED_SIGN_UP_TO_SHAREHOLDER);

    return $shareholder;
  }

  /**
   * {@inheritdoc}
   */
  public function testPaidInFull(WebformSubmissionInterface $submission, $account) {
    $shareholder_shares_key = $this->configuration['shares_key'];
    $data = $submission->getData();

    $share_price = ShareholderShares::getSharePrice(
      $data[$shareholder_shares_key]['share_issue'],
      $data[$shareholder_shares_key]['share_quantity']
    );

    if ($account->getCreditBalance() >= $share_price['total']) {
      return $account->getCreditBalanceDate($share_price['total']);
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function createOrGetShareTransaction(WebformSubmissionInterface $submission, $shareholder, $account, $payment_date) {
    $transaction_data = $this->getIssueAndQuantity($submission);

    $share_transaction = \Drupal::service('shareholder_register.default')->createIssueTransaction(
      $shareholder,
      $transaction_data['issue'],
      $transaction_data['quantity'],
      $account,
      $payment_date);

    $this->mapSubmissionToEntity(
      $this->configuration['share_transaction_mapping'],
      $submission,
      $share_transaction);

    $share_transaction->save();

    // @todo refactor to workflow transition.
    if ($account->getState() === 'open') {
      $account->setState('pending');
    }
    $account->save();

    return $share_transaction;
  }

  /**
   * {@inheritdoc}
   */
  public function performMapping(WebformSubmissionInterface $submission, AccountingAccount $account, $test_balance = TRUE) {
    $date = $this->testPaidInFull($submission, $account);
    if (!$test_balance || $date) {
      $shareholder = $this->createOrGetShareholder($submission);
      $transaction = $this->createOrGetShareTransaction($submission, $shareholder, $account, $date);
      if (\Drupal::config('shareholder_register.settings')->get('automatic_validation')) {
        try {
          $transaction->actionValidate($date);
        }
        catch (ShareholderRegisterException $e) {
          \Drupal::messenger()->addError(
            $this->t('Could not automatically validate share transation:') . ' ' .
            $this->t($e->getMessage(), $e->getMsgPlaceholders()));
        }

        // Reload account, because actionValidate might have changed it's state.
        $account = AccountingAccount::load($account->id());
      }
      if ($account->hasField('field_shareholder')) {
        $account->set('field_shareholder', $shareholder->id());
        $account->save();
      }
      return $transaction;
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'shareholder_number_key' => '',
      'shareholder_type_key' => '',
      'save_shareholder_id_key' => '',
      'shares_key' => '',
      'shareholder_mapping' => '',
      'share_transaction_mapping' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['additional'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Mapping settings'),
    ];
    $form['additional']['shareholder_number_key'] = [
      '#title' => 'Shareholder number element',
      '#type' => 'textfield',
      '#parents' => ['settings', 'shareholder_number_key'],
      '#default_value' => $this->configuration['shareholder_number_key'],
    ];
    $form['additional']['save_shareholder_id_key'] = [
      '#title' => 'Save account ID key',
      '#type' => 'textfield',
      '#parents' => ['settings', 'save_shareholder_id_key'],
      '#default_value' => $this->configuration['save_shareholder_id_key'],
    ];
    $form['additional']['shareholder_type_key'] = [
      '#title' => 'Shareholder type element',
      '#type' => 'textfield',
      '#parents' => ['settings', 'shareholder_type_key'],
      '#default_value' => $this->configuration['shareholder_type_key'],
    ];
    $form['additional']['shareholder_shares_key'] = [
      '#title' => 'Shareholder shares element',
      '#type' => 'textfield',
      '#parents' => ['settings', 'shares_key'],
      '#default_value' => $this->configuration['shares_key'],
    ];
    $form['additional']['shareholder_mapping'] = [
      '#title' => 'Shareholder Mapping',
      '#type' => 'textarea',
      '#parents' => ['settings', 'shareholder_mapping'],
      '#default_value' => $this->configuration['shareholder_mapping'],
    ];
    $form['additional']['share_transaction_mapping'] = [
      '#title' => 'Share Transaction Mapping',
      '#type' => 'textarea',
      '#parents' => ['settings', 'share_transaction_mapping'],
      '#default_value' => $this->configuration['share_transaction_mapping'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    parent::applyFormStateToConfiguration($form_state);
  }

}
