<?php

namespace Drupal\dsr;

use Composer\InstalledVersions;

use Symfony\Component\Validator\Constraints\Iban;
use Symfony\Component\Validator\Validation;

use Drupal\Core\Extension\ModuleHandlerInterface;

use Drupal\shareholder_register\Entity\ShareType;
use Drupal\shareholder_register\Entity\ShareIssue;
use Drupal\shareholder_register\Entity\Shareholder;
use Drupal\shareholder_register\Entity\ShareTransaction;
use Drupal\shareholder_register\Entity\ShareTransactionGroup;
use Drupal\shareholder_register_dividend\Entity\DividendChoice;

use Drupal\accounting\Entity\AccountingAccount;
use Drupal\accounting\Entity\AccountingJournal;

use Drupal\banking\Entity\BankAccount;
use Drupal\payment\Entity\PaymentMethodConfiguration;

use Drupal\shareholder_register\Exception\ShareholderRegisterException;

use Drupal\taxonomy\Entity\Term;

/**
 * Class DsrDrushService.
 */
class DsrDrushService implements DsrDrushServiceInterface {

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs a new DsrDrushService object.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  public function __construct(ModuleHandlerInterface $module_handler) {
    $this->moduleHandler = $module_handler;
  }

  /**
   * Import read csv and call callback.
   */
  protected function importFromCsv($import_file, $callback, &$context) {
    if (($handle = fopen($import_file, "r")) !== FALSE) {
      $headers = FALSE;

      while (($data = fgetcsv($handle, 1000, ",")) !== FALSE) {
        $data = array_map('trim', $data);
        if (!$headers) {
          $headers = $data;

          // Fix headers.
          $replacement = [
            'shareholder_number' => '*shareholder_id|shareholder|number',
          ];
          $headers = array_map(function ($v) use ($replacement) {
            return isset($replacement[$v]) ? $replacement[$v] : $v;
          }, $headers);

          continue;
        }

        $line_data = [];
        foreach ($headers as $k => $header) {
          $header_first_char = substr($header, 0, 1);
          if (!ctype_alpha($header_first_char) && $header_first_char !== '*') {
            continue;
          }

          if ($header_first_char == '*') {
            $header_parts = explode('|', substr($header, 1));
            if (count($header_parts) != 3) {
              continue;
            }
            $entities = \Drupal::entityTypeManager()->getStorage(
              $header_parts[1])->loadByProperties([
                $header_parts[2] => $data[$k],
              ]
              );
            $header = $header_parts[0];
            if (count($entities)) {
              $data[$k] = reset($entities)->id();
            }
            else {
              $data[$k] = FALSE;
            }
          }

          if (strpos($header, '.') === FALSE) {
            $line_data[$header] = $data[$k];
          }
          else {
            $el = explode(".", $header);
            if (count($el) == 2) {
              $line_data[$el[0]][$el[1]] = $data[$k];
            }
            elseif (count($el) == 3) {
              $line_data[$el[0]][$el[1]][$el[2]] = $data[$k];
            }
          }
        }

        if (count($line_data)) {
          $params = [$line_data, &$context];
          call_user_func_array($callback, $params);
        }
      }
    }
  }

  /**
   * Import create shareholder.
   */
  protected static function createShareholder($data, &$context) {
    $s = Shareholder::create(array_filter($data));
    $s->save();
    return $s;
  }

  /**
   * Import create share transaction.
   */
  protected static function createShareTransactionGroup($data, &$context) {
    $g = ShareTransactionGroup::create(array_filter($data));
    $g->save();
    return $g;
  }

  /**
   * Import create share transaction.
   */
  protected static function createShareTransaction($data, &$context) {
    if (isset($data['transaction_group']) && !empty($data['transaction_group'])) {
      if (!isset($context['groups'][$data['transaction_group']])) {
        $context['groups'][$data['transaction_group']] = static::createShareTransactionGroup([
          'type' => 'transfer',
        ], $context)->id();
      }
      $data['transaction_group'] = $context['groups'][$data['transaction_group']];
    }
    $t = ShareTransaction::create(array_filter($data));
    $t->save();
    return $t;
  }

  /**
   * {@inheritdoc}
   */
  public function importShareholderRegister($directory) {
    $context = [];

    if (file_exists($directory . DIRECTORY_SEPARATOR . 'accounting_accounts.csv')) {
      $this->importFromCsv($directory . DIRECTORY_SEPARATOR . 'accounting_accounts.csv', function ($data, &$context) {
        $a = AccountingAccount::create(array_filter($data));
        $a->save();
        return $a;
      }, $context);
    }
    if (file_exists($directory . DIRECTORY_SEPARATOR . 'accounting_journals.csv')) {
      $this->importFromCsv($directory . DIRECTORY_SEPARATOR . 'accounting_journals.csv', function ($data, &$context) {
        $a = AccountingJournal::create(array_filter($data));
        $a->save();
        return $a;
      }, $context);
    }
    if (file_exists($directory . DIRECTORY_SEPARATOR . 'bank_accounts.csv')) {
      $this->importFromCsv($directory . DIRECTORY_SEPARATOR . 'bank_accounts.csv', function ($data, &$context) {
        $a = BankAccount::create(array_filter($data));
        $a->save();
        return $a;
      }, $context);
    }
    if (file_exists($directory . DIRECTORY_SEPARATOR . 'projects.csv')) {
      $this->importFromCsv($directory . DIRECTORY_SEPARATOR . 'projects.csv', function ($data, &$context) {
        $a = Term::create(array_filter($data));
        $a->save();
        return $a;
      }, $context);
    }
    if (file_exists($directory . DIRECTORY_SEPARATOR . 'share_types.csv')) {
      $this->importFromCsv($directory . DIRECTORY_SEPARATOR . 'share_types.csv', function ($data, &$context) {
        $a = ShareType::create(array_filter($data));
        $a->save();
        return $a;
      }, $context);
    }
    if (file_exists($directory . DIRECTORY_SEPARATOR . 'share_issues.csv')) {
      $this->importFromCsv($directory . DIRECTORY_SEPARATOR . 'share_issues.csv', function ($data, &$context) {
        $a = ShareIssue::create(array_filter($data));
        $a->save();
        return $a;
      }, $context);
    }
    if (file_exists($directory . DIRECTORY_SEPARATOR . 'dividend_choices.csv')) {
      $this->importFromCsv($directory . DIRECTORY_SEPARATOR . 'dividend_choices.csv', function ($data, &$context) {
        $a = DividendChoice::create(array_filter($data));
        foreach($data['third_party'] as $module => $values) {
          foreach ($values as $key => $value) {
            $a->setThirdPartySetting($module, $key, $value);
          }
        }
        $a->save();
        return $a;
      }, $context);
    }

    $this->importFromCsv($directory . DIRECTORY_SEPARATOR . 'shareholders.csv', [$this, 'createShareholder'], $context);
    $this->importFromCsv($directory . DIRECTORY_SEPARATOR . 'share_transactions.csv', [$this, 'createShareTransaction'], $context);

    static::postImportConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function importShareholderRegisterValidate() {
    $t_ids = \Drupal::EntityQuery('share_transaction')
      ->accessCheck(FALSE)
      ->condition('state', 'draft')
      ->sort('payment_date', 'ASC')
      ->sort('quantity', 'DESC')
      ->range(0, 100)
      ->execute();

    if (!count($t_ids)) {
      throw new \Exception("All transactions are validated!");
    }

    foreach (ShareTransaction::loadMultiple($t_ids) as $t) {
      if ($t->getState() !== 'valid') {
        try {
          $t->actionValidate($t->getPaymentDate());
        }
        catch (ShareholderRegisterException $e) {
          error_log('Could not validate transation:' . t(
              $e->getMessage(), $e->getMsgPlaceholders()));
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function copyAccountNumberToIbanBic($entity, $account_field, $iban_field) {
    $storage = \Drupal::entityTypeManager()->getStorage($entity);
    $validator = Validation::createValidator();
    foreach ($storage->loadMultiple() as $e) {
      if (!$e->hasField($account_field) || !$e->hasField($iban_field)) {
        error_log("{$entity} {$e->id()} doesn't have all fields!");
        continue;
      }
      if (trim($e->get($account_field)->value) &&
        !count($validator->validate($e->get($account_field)->value, [new Iban()]))) {
        $iban = $e->get($account_field)->value;
        $bic = '';

        $json = \Drupal::service('iban_bic_field.default')->validateIbanBic($iban, $bic);
        if (isset($json['bic'])) {
          $bic = $json['bic'];
        }
        $e->get($iban_field)->iban = iban_to_human_format($iban);
        $e->get($iban_field)->bic = iban_to_machine_format($bic);
        $e->save();
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  private static function accountByCode($code) {
    $a = reset(\Drupal::entityTypeManager()->getStorage(
           'accounting_account')->loadByProperties(['code' => $code]));
    if ($a) {
      return $a->id();
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  private static function journalByName($name) {
    $a = reset(\Drupal::entityTypeManager()->getStorage(
           'accounting_journal')->loadByProperties(['name' => $name]));
    if ($a) {
      return $a->id();
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public static function postImportConfiguration() {
    $payment_method = PaymentMethodConfiguration::load('mollie');
    if ($payment_method) {
      $payment_method
        ->setThirdPartySetting('accounting_payment', 'account',
          static::accountByCode('550001'))
        ->setThirdPartySetting('accounting_payment', 'journal',
          static::journalByName('mollie_payment_prod'))
        ->save();
    }

    \Drupal::service('config.factory')->getEditable('accounting_deposit.settings')
      ->set('manual_deposit_account',
        static::accountByCode('570000'))
      ->set('manual_deposit_journal',
        static::journalByName('Manual payment'))
      ->save();

    \Drupal::service('config.factory')->getEditable('shareholder_register_accounting.settings')
      ->set('share_transaction_account',
        static::accountByCode('410000'))
      ->set('share_transaction_journal',
        static::journalByName('Diverse'))
      ->set('share_value_account',
        static::accountByCode('100000'))
      ->save();

    \Drupal::service('config.factory')->getEditable('sr_accounting_dividend.settings')
      ->set('dividend_account',
        static::accountByCode('694000'))
      ->set('dividend_journal',
        static::journalByName('Diverse'))
      ->set('dividend_tax_account',
        static::accountByCode('453000'))
      ->save();
  }

  /**
   * {@inheritdoc}
   */
  public function getVersion() {
    $version = [];
    if (InstalledVersions::isInstalled('drupal/dsr')) {
      $version['dsr'] = [
        'name' => "DSR",
        'version' => InstalledVersions::getVersion('drupal/dsr'),
      ];
    }
    $this->moduleHandler->alter('dsr_version', $version);
    return $version;
  }
}
