<?php

namespace Drupal\shareholder_register_dividend;

use Drupal\shareholder_register\Entity\Shareholder;

/**
 * Class DividendService.
 */
class DividendService implements DividendServiceInterface {

  /**
   * Constructs a new DividendService object.
   */
  public function __construct() {

  }

  /**
   * {@inheritdoc}
   */
  public static function getDividendAllocationDefinition($type) {
    $definition = \Drupal::service('plugin.manager.dividend_allocation_definitions')
      ->createInstance($type, []);
    if (!$type) {
      throw new \Exception("Plugin '{$type}' undefined!");
    }
    return $definition;
  }

  /**
   * {@inheritdoc}
   */
  public static function getDividendTaxDefinition($type) {
    $definition = \Drupal::service('plugin.manager.dividend_tax_definitions')
      ->createInstance($type, []);
    if (!$type) {
      throw new \Exception("Plugin '{$type}' undefined!");
    }
    return $definition;
  }

  /**
   * {@inheritdoc}
   */
  public static function initDividendBatch($dividend_config, &$context) {
    $context['results']['details'] = [];
    $context['results']['totals'] = [];
    $context['results']['config'] = $dividend_config;
    $context['results']['base'] = DividendService::getDividendAllocationDefinition(
      $dividend_config['allocation'])->getDividendBase($dividend_config);
  }

  /**
   * {@inheritdoc}
   */
  public static function dividendBatch($dividend_config, &$context) {
    if (!isset($context['sandbox']['shareholder_ids'])) {
      $shareholder_ids = \Drupal::entityQuery('shareholder')
        ->condition('state', 'valid')
        ->execute();

      $context['sandbox']['shareholder_ids'] = $shareholder_ids;
      $context['sandbox']['shareholder_ids_count'] = count($context['sandbox']['shareholder_ids']);
      $context['results']['totals'] = [];
    }
    $allocation_plugin = DividendService::getDividendAllocationDefinition($dividend_config['allocation']);
    $tax_plugin = DividendService::getDividendTaxDefinition($dividend_config['tax']);

    for ($i = 0; $i < 100; $i++) {
      if ($shareholder_id = array_pop($context['sandbox']['shareholder_ids'])) {
        $shareholder = Shareholder::load($shareholder_id);

        if ($shareholder->hasField('field_no_dividend') && $shareholder->get('field_no_dividend')->value) {
          continue;
        }

        $dividend_details = $allocation_plugin->computeDividend(
          $shareholder, $dividend_config);

        $tax_plugin->computeDividendTaxRate(
          $shareholder,
          $dividend_details,
          $dividend_config
        );

        $context['results']['details'] = array_merge(
          $context['results']['details'],
          $dividend_details
        );

        foreach ($dividend_details as $details) {
          $hash = call_user_func($dividend_config['total_hash'], $details);
          if (!array_key_exists($hash, $context['results']['totals'])) {
            $context['results']['totals'][$hash] = [
              'fraction' => $details->fraction,
              'shareholder_id' => $details->shareholder_id,
              'grouped' => [],
            ];
          }
          else {
            $context['results']['totals'][$hash]['fraction'] += $details->fraction;
          }

          if (!isset($context['results']['totals'][$hash]['grouped'][$details->tax])) {
            $context['results']['totals'][$hash]['grouped'][$details->tax] = [
              'fraction' => $details->fraction,
              'tax' => $details->tax,
              'taxcode' => [$details->taxcode],
              'details' => [(array) $details],
            ];
          }
          else {
            if (!in_array($details->taxcode, $context['results']['totals'][$hash]['grouped'][$details->tax]['taxcode'])) {
              $context['results']['totals'][$hash]['grouped'][$details->tax]['taxcode'][] = $details->taxcode;
            }
            $context['results']['totals'][$hash]['grouped'][$details->tax]['fraction'] += $details->fraction;
            $context['results']['totals'][$hash]['grouped'][$details->tax]['details'][] = (array) $details;
          }
        }
      }
    }

    if ($context['sandbox']['shareholder_ids_count'] > 0) {
      $context['finished'] = ($context['sandbox']['shareholder_ids_count'] - count($context['sandbox']['shareholder_ids'])) / $context['sandbox']['shareholder_ids_count'];
    }
    else {
      $context['finished'] = 1;
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function dividendTotalHashWithTaxDetails($details) {
    return "{$details->shareholder_id}-{$details->tax}";
  }

  /**
   * {@inheritdoc}
   */
  public static function dividendTotalHash($details) {
    return "{$details->shareholder_id}";
  }

  /**
   * {@inheritdoc}
   */
  public function getDividendAllocationBatch($dividend_config) {
    $batch = array(
      'title' => t('Computing dividend'),
      'operations' => [],
    );
    $batch['operations'][] = [
      '\Drupal\shareholder_register_dividend\DividendService::initDividendBatch',
      [$dividend_config],
    ];
    $batch['operations'][] = [
      '\Drupal\shareholder_register_dividend\DividendService::dividendBatch',
      [$dividend_config],
    ];

    return $batch;
  }

  /**
   * {@inheritdoc}
   */
  public function registerDividend($shareholder_number, $gross, $net, $distribution) {

  }

  /**
   * {@inheritdoc}
   */
  public function getDividendContext($entity, $distribution) {
    $gross = 0;
    $net = 0;
    $tax = 0;
    $dividends = [];

    foreach ($entity->get('dividends')->referencedEntities() as $d) {
      if ($d->get('distribution_id')->target_id == $distribution) {
        $dividends[] = $d;
        $gross = bcadd($gross, $d->get('gross')->value, 2);
        $net = bcadd($net, $d->get('net')->value, 2);
      }
    }
    $tax = bcsub($gross, $net, 2);

    if ($tax > 0 || $gross > 0) {
      $context = [
        'distribution_id' => $distribution,
        'shareholder' => $entity,
        'dividends' => $dividends,
        'gross' => $gross,
        'net' => $net,
        'tax' => $tax,
      ];
    }

    return $context;
  }

  /**
   * {@inheritdoc}
   */
  public function getTotalDividends($distribution) {
    $totals = [];
    foreach ($distribution->get('shareholder_dividends')->referencedEntities() as $shareholder_dividend) {
      $shareholder = $shareholder_dividend->get('shareholder_id')->entity;
      if (!isset($totals[$shareholder->id()])) {
        $totals[$shareholder->id()] = [
          'shareholder' => $shareholder,
          'gross' => 0,
          'net' => 0,
          'tax' => 0,
        ];
      }
      $gross = $shareholder_dividend->get('gross')->value;
      $net = $shareholder_dividend->get('net')->value;
      $tax = bcsub($gross, $net, 2);

      $totals[$shareholder->id()]['net'] = bcadd($totals[$shareholder->id()]['net'], $net, 2);
      $totals[$shareholder->id()]['gross'] = bcadd($totals[$shareholder->id()]['gross'], $gross, 2);
      $totals[$shareholder->id()]['tax'] = bcadd($totals[$shareholder->id()]['tax'], $tax, 2);
    }
    return $totals;
  }

  /**
   * {@inheritdoc}
   */
  public function sendDividendMail($entity, $distribution) {
    $context = $this->getDividendContext($entity, $distribution);

    if ($context['gross'] > 0 || $context['tax'] > 0) {
      $langcode = $entity->getPreferredLangcode();
      $mailManager = \Drupal::service('plugin.manager.mail');

      $email = $entity->get('mail')->value;
      if (strpos($email, '@') === FALSE) {
        $email = \Drupal::config('system.site')->get('mail');
      }
      $params = [
        'context' => $context,
      ];
      $result = $mailManager->mail('shareholder_register_dividend', 'mail_dividend', $email, $langcode, $params, NULL, TRUE);
    }
    else {
      error_log("Not sending dividend mail for {$entity->getNumber()}: no dividend!");
    }
  }

}
