<?php

namespace Drupal\shareholder_register_dividend\Entity;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\user\UserInterface;

use Drupal\shareholder_register_dividend\Exception\ShareholderRegisterDividendException;
use Drupal\shareholder_register_dividend\Plugin\Field\FieldType\DistributionTotalItemList;

/**
 * Defines the Dividend distribution entity.
 *
 * @ingroup shareholder_register_dividend
 *
 * @ContentEntityType(
 *   id = "dividend_distribution",
 *   label = @Translation("Dividend distribution"),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\shareholder_register_dividend\DividendDistributionListBuilder",
 *     "views_data" = "Drupal\shareholder_register_dividend\Entity\DividendDistributionViewsData",
 *
 *     "form" = {
 *       "default" = "Drupal\shareholder_register_dividend\Form\DividendDistributionForm",
 *       "add" = "Drupal\shareholder_register_dividend\Form\DividendDistributionForm",
 *       "edit" = "Drupal\shareholder_register_dividend\Form\DividendDistributionForm",
 *       "delete" = "Drupal\shareholder_register_dividend\Form\DividendDistributionDeleteForm",
 *     },
 *     "access" = "Drupal\shareholder_register_dividend\DividendDistributionAccessControlHandler",
 *     "route_provider" = {
 *       "html" = "Drupal\shareholder_register_dividend\DividendDistributionHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "dividend_distribution",
 *   admin_permission = "administer dividend distribution entities",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "name",
 *     "uuid" = "uuid",
 *     "uid" = "user_id",
 *   },
 *   links = {
 *     "canonical" = "/admin/shareholder_register/tools/shareholder_dividend/dividend_distribution/{dividend_distribution}",
 *     "add-form" = "/admin/shareholder_register/tools/shareholder_dividend/dividend_distribution/add",
 *     "edit-form" = "/admin/shareholder_register/tools/shareholder_dividend/dividend_distribution/{dividend_distribution}/edit",
 *     "delete-form" = "/admin/shareholder_register/tools/shareholder_dividend/dividend_distribution/{dividend_distribution}/delete",
 *     "collection" = "/admin/shareholder_register/tools/shareholder_dividend/dividend_distribution",
 *   },
 *   field_ui_base_route = "dividend_distribution.settings"
 * )
 */
class DividendDistribution extends ContentEntityBase implements DividendDistributionInterface {

  use EntityChangedTrait;

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage_controller, array &$values) {
    parent::preCreate($storage_controller, $values);
    $values += [
      'user_id' => \Drupal::currentUser()->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function delete() {
    if ($this->getState() == 'valid') {
      throw new ShareholderRegisterDividendException(
        "The dividend distribution is not in state 'draft'!");
    }
    else {
      foreach ($this->get('shareholder_dividends')->referencedEntities() as $dividend) {
        $dividend->delete();
      }
      return parent::delete();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getName() {
    return $this->get('name')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setName($name) {
    $this->set('name', $name);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getDate() {
    return $this->get('date')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setDate($date) {
    return $this->set('date', $date);
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime($timestamp) {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwner() {
    return $this->get('user_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwnerId() {
    return $this->get('user_id')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwnerId($uid) {
    $this->set('user_id', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwner(UserInterface $account) {
    $this->set('user_id', $account->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getState() {
    return $this->get('state')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setState($name) {
    $this->set('state', $name);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['user_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Authored by'))
      ->setDescription(t('The user ID of author of the Dividend distribution entity.'))
      ->setRevisionable(TRUE)
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setTranslatable(TRUE)
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Name'))
      ->setDescription(t('The name of the Dividend distribution entity.'))
      ->setSettings([
        'max_length' => 50,
        'text_processing' => 0,
        'wkf-editable' => ['state' => ['draft']],
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 1,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 1,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    $fields['date'] = BaseFieldDefinition::create('datetime')
      ->setLabel(t('Dividend Date'))
      ->setDescription(t('The Date of the dividend distribution.'))
      ->setSettings([
        'datetime_type' => 'date',
        'wkf-editable' => ['state' => ['draft']],
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 2,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 2,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['total_gross'] = BaseFieldDefinition::create('decimal')
      ->setLabel(t('Total Gross'))
      ->setComputed(TRUE)
      ->setSettings([
        'sum_field' => 'gross',
      ])
      ->setClass(DistributionTotalItemList::class)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'weight' => 3,
      ]);

    $fields['total_net'] = BaseFieldDefinition::create('decimal')
      ->setLabel(t('Total Net'))
      ->setComputed(TRUE)
      ->setSettings([
        'sum_field' => 'net',
      ])
      ->setClass(DistributionTotalItemList::class)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'weight' => 5,
      ]);

    $fields['shareholder_dividends'] = BaseFieldDefinition::create('shareholder_dividends')
      ->setLabel(t('Shareholder Dividends'))
      ->setComputed(TRUE)
      ->setSettings([
        'target_type' => 'shareholder_dividend',
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'distribution_dividends_formatter',
        'label' => 'above',
        'weight' => 6,
      ]);

    $fields['state'] = BaseFieldDefinition::create('simple_workflow_state_field')
      ->setLabel(t('State'))
      ->setDescription(t('State of the Dividend Distribution.'))
      ->setRevisionable(TRUE)
      ->setDefaultValue('draft');

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the entity was last edited.'));

    return $fields;
  }

}
