<?php

namespace Drupal\shareholder_register_dividend\Entity;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\user\UserInterface;

/**
 * Defines the Shareholder dividend entity.
 *
 * @ingroup shareholder_register_dividend
 *
 * @ContentEntityType(
 *   id = "shareholder_dividend",
 *   label = @Translation("Shareholder dividend"),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\shareholder_register_dividend\ShareholderDividendListBuilder",
 *     "views_data" = "Drupal\shareholder_register_dividend\Entity\ShareholderDividendViewsData",
 *
 *     "form" = {
 *       "default" = "Drupal\shareholder_register_dividend\Form\ShareholderDividendForm",
 *       "add" = "Drupal\shareholder_register_dividend\Form\ShareholderDividendForm",
 *       "edit" = "Drupal\shareholder_register_dividend\Form\ShareholderDividendForm",
 *       "delete" = "Drupal\shareholder_register_dividend\Form\ShareholderDividendDeleteForm",
 *     },
 *     "access" = "Drupal\shareholder_register_dividend\ShareholderDividendAccessControlHandler",
 *     "route_provider" = {
 *       "html" = "Drupal\shareholder_register_dividend\ShareholderDividendHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "shareholder_dividend",
 *   admin_permission = "administer shareholder dividend entities",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "name",
 *     "uuid" = "uuid",
 *     "uid" = "user_id",
 *   },
 *   links = {
 *     "canonical" = "/admin/shareholder_register/tools/shareholder_dividend/{shareholder_dividend}",
 *     "add-form" = "/admin/shareholder_register/tools/shareholder_dividend/add",
 *     "edit-form" = "/admin/shareholder_register/tools/shareholder_dividend/{shareholder_dividend}/edit",
 *     "delete-form" = "/admin/shareholder_register/tools/shareholder_dividend/{shareholder_dividend}/delete",
 *     "collection" = "/admin/shareholder_register/tools/shareholder_dividend",
 *   },
 *   field_ui_base_route = "shareholder_dividend.settings"
 * )
 */
class ShareholderDividend extends ContentEntityBase implements ShareholderDividendInterface {

  use EntityChangedTrait;

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage_controller, array &$values) {
    parent::preCreate($storage_controller, $values);
    $values += [
      'user_id' => \Drupal::currentUser()->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getName() {
    return $this->get('name')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setName($name) {
    $this->set('name', $name);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getShareholder() {
    return $this->get('shareholder_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime($timestamp) {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwner() {
    return $this->get('user_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwnerId() {
    return $this->get('user_id')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwnerId($uid) {
    $this->set('user_id', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwner(UserInterface $account) {
    $this->set('user_id', $account->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getData($key, $default = NULL) {
    $data = [];
    if (!$this->get('extra')->isEmpty()) {
      $data = $this->get('extra')->first()->getValue();
    }
    return isset($data[$key]) ? $data[$key] : $default;
  }

  /**
   * {@inheritdoc}
   */
  public function setData($key, $value) {
    $this->get('extra')->__set($key, $value);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getGross() {
    return $this->get('gross')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getNet() {
    return $this->get('net')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getTax() {
    return bcsub(
      $this->getGross(),
      $this->getNet(),
      2);
  }

  /**
   * {@inheritdoc}
   */
  public function getDistribution() {
    return $this->get('distribution_id')->entity;
  }


  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['user_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Authored by'))
      ->setDescription(t('The user ID of author of the Shareholder dividend entity.'))
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['distribution_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Distribution'))
      ->setDescription(t('The distribution of the Shareholder dividend entity.'))
      ->setSettings([
        'wkf-editable' => ['state' => ['draft']],
        'target_type' => 'dividend_distribution',
        'handler' => 'default',
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'author',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 5,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['shareholder_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Shareholder'))
      ->setDescription(t('The Shareholder.'))
      ->setSettings([
        'target_type' => 'shareholder',
        'default_value' => 0,
        'wkf-editable' => ['state' => ['draft']],
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'author',
        'weight' => 2,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 2,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Name'))
      ->setDescription(t('The name of the Shareholder dividend entity.'))
      ->setSettings([
        'max_length' => 50,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    $fields['gross'] = BaseFieldDefinition::create('decimal')
      ->setLabel(t('Gross Amount'))
      ->setDescription(t('The Gross amount of the dividend.'))
      ->setSettings([
        'wkf-editable' => ['state' => ['draft']],
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'number_decimal',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'number',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['net'] = BaseFieldDefinition::create('decimal')
      ->setLabel(t('Net Amount'))
      ->setDescription(t('The Net Amount of the Dividend.'))
      ->setSettings([
        'wkf-editable' => ['state' => ['draft']],
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'number_decimal',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'number',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['state'] = BaseFieldDefinition::create('string')
      ->setLabel(t('State'))
      ->setDescription(t('State of the Shareholder Dividend.'))
      ->setRevisionable(TRUE)
      ->setDefaultValue('draft');

    $fields['extra'] = BaseFieldDefinition::create('map')
      ->setLabel(t('Extra'))
      ->setDescription(t('Extra data for this Dividend.'));

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the entity was last edited.'));

    return $fields;
  }

}
