<?php

namespace Drupal\shareholder_register_dividend\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;

use Drupal\Core\Url;
use Drupal\Core\Link;

use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Reader\Xlsx;

use Drupal\shareholder_register\Entity\Shareholder;
use Drupal\shareholder_register\Entity\Share;

use Drupal\shareholder_register_dividend\Entity\DividendDistribution;
use Drupal\shareholder_register_dividend\Entity\ShareholderDividend;

/**
 * Class DividendForm.
 */
class DividendForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public static function registerDividend($distribution, &$context) {
    $internal_precision = 4;

    $config = $context['results']['config'];
    if (!isset($context['sandbox']['to_write'])) {
      $context['sandbox']['to_write'] = $context['results']['totals'];
      $dist = DividendDistribution::create([
        'name' => $distribution,
      ]);
      $dist->save();
      $context['sandbox']['distribution'] = $dist;
    }

    $formatterService = \Drupal::service('shareholder_register.formatter');

    for ($i = 0; $i < 100; $i++) {
      if ($line = array_pop($context['sandbox']['to_write'])) {
        $gross = '0.0';
        $tax = '0.0';

        foreach ($line['grouped'] as $tax_rate => &$data) {
          $fraction = $data['fraction'];

          $sub_gross = $formatterService->round(
            bcdiv(
              bcmul(
                $fraction,
                $config['dividend'],
                $internal_precision
              ),
              $context['results']['base'],
              $internal_precision + 1
            ),
            2
          );

          $gross = bcadd(
            $gross,
            $sub_gross,
            2
          );

          $sub_tax = $formatterService->round(
            bcdiv(
              bcmul(
                $sub_gross,
                $tax_rate,
                2),
              100,
              2 + 1
            ),
            2
          );

          $tax = bcadd(
            $tax,
            $sub_tax,
            2
          );

          $data['gross'] = $sub_gross;
          $data['tax'] = $sub_tax;
          $data['net'] = bcsub(
            $sub_gross,
            $sub_tax,
            2
          );
        }

        $div = ShareholderDividend::create([
          'distribution_id' => $context['sandbox']['distribution']->id(),
          'shareholder_id' => $line['shareholder_id'],
          'gross' => $gross,
          'net' => bcsub($gross, $tax, 2),
        ]);
        $div->setData('details', $line);
        $div->save();
      }
    }

    if (count($context['results']['totals'])) {
      $context['finished'] = (count($context['results']['totals']) - count($context['sandbox']['to_write'])) / count($context['results']['totals']);
    }
    else {
      $context['finished'] = 1;
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function finishRegisterDividend($success, $results, $operations) {
    drupal_set_message(t(
        'Dividend registered!',
        []
    ));
  }

  /**
   * {@inheritdoc}
   */
  public static function writeExportDividendDetails(&$context) {
    if (!isset($context['results']['output'])) {
      $output = file_save_data('', 'private://dividend.xlsx');
      $output->setTemporary();
      $output->save();

      $context['results']['output'] = $output;

      $spreadsheet = new Spreadsheet();

      // Set document properties.
      $spreadsheet->getProperties()->setCreator('DSR')
        ->setLastModifiedBy(t('DSR'))
        ->setTitle(t('DSR Dividend simulation'))
        ->setSubject(t('DSR Dividend simulation'))
        ->setDescription(t('DSR Dividend simulation'))
        ->setCategory(t('DSR Dividend'));
    }
    else {
      $reader = new Xlsx();
      $reader->setReadDataOnly(TRUE);
      $spreadsheet = $reader->load(drupal_realpath($context['results']['output']->getFileUri()));
    }

    if (!isset($context['sandbox']['to_write'])) {
      $config = $context['results']['config'];
      $spreadsheet->createSheet();
      $sheet = $spreadsheet->setActiveSheetIndex(1);

      $sheet->setCellValue("A1", t('Dividend for period @begin - @end',
      [
        '@begin' => $config['start_date'],
        '@end' => $config['end_date'],
      ]));
      $sheet->setCellValue("A3", t('Dividend'));
      $sheet->setCellValue("B3", $config['dividend']);
      $sheet->setCellValue("C3", $context['results']['base']);
      $sheet->setCellValue("A4", t('Total'));

      $sheet->setCellValue("A6", t('Shareholder Number'));
      $sheet->setCellValue("B6", t('Shareholder Name'));
      $sheet->setCellValue("C6", t('Shares (fraction)'));
      $sheet->setCellValue("D6", t('Tax Rate'));
      $sheet->setCellValue("E6", t('Gross'));
      $sheet->setCellValue("F6", t('Tax Amount'));
      $sheet->setCellValue("G6", t('Dividend'));
      $sheet->setCellValue("H6", t('Details'));

      $context['sandbox']['row'] = 7;
      $context['sandbox']['to_write'] = $context['results']['details'];
    }
    else {
      $sheet = $spreadsheet->setActiveSheetIndex(1);
      $sheet = $spreadsheet->getActiveSheet();
    }

    for ($i = 0; $i < 5000; $i++) {
      if ($line = array_pop($context['sandbox']['to_write'])) {
        $row = $context['sandbox']['row'];

        $shareholder = Shareholder::load($line->shareholder_id);
        $share = Share::load($line->share_id);

        $sheet->setCellValue("A{$row}", $shareholder->getNumber());
        $sheet->setCellValue("B{$row}", $shareholder->getName());
        $sheet->setCellValue("C{$row}", $line->fraction);
        $sheet->setCellValue("D{$row}", $line->tax);
        $sheet->setCellValue("E{$row}", "=ROUND(C{$row}*B3/C3, 2)");
        $sheet->setCellValue("F{$row}", "=ROUND(E{$row}*(D{$row}/100), 2)");
        $sheet->setCellValue("G{$row}", "=E{$row}-F{$row}");
        $sheet->setCellValue("H{$row}", "{$share->getName()}: {$line->start_date} - {$line->end_date}");

        $sheet->setCellValue("B4", "=SUM(E7:E{$row})");

        $context['sandbox']['row']++;
      }
    }

    $writer = IOFactory::createWriter($spreadsheet, 'Xlsx');
    $writer->setPreCalculateFormulas(FALSE);
    $writer->save(drupal_realpath($context['results']['output']->getFileUri()));
    $context['results']['output']->save();

    if (count($context['results']['details'])) {
      $context['finished'] = (count($context['results']['details']) - count($context['sandbox']['to_write'])) / count($context['results']['details']);
    }
    else {
      $context['finished'] = 1;
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function writeExportDividend(&$context) {
    if (!isset($context['results']['output'])) {
      $output = file_save_data('', 'private://dividend.xlsx');
      $output->setTemporary();
      $output->save();

      $context['results']['output'] = $output;

      $spreadsheet = new Spreadsheet();

      // Set document properties.
      $spreadsheet->getProperties()->setCreator('DSR')
        ->setLastModifiedBy(t('DSR'))
        ->setTitle(t('DSR Dividend simulation'))
        ->setSubject(t('DSR Dividend simulation'))
        ->setDescription(t('DSR Dividend simulation'))
        ->setCategory(t('DSR Dividend'));
    }
    else {
      $reader = new Xlsx();
      $reader->setReadDataOnly(TRUE);
      $spreadsheet = $reader->load(drupal_realpath($context['results']['output']->getFileUri()));
    }

    if (!isset($context['sandbox']['to_write'])) {
      $config = $context['results']['config'];
      $sheet = $spreadsheet->setActiveSheetIndex(0);

      $sheet->setCellValue("A1", t('Dividend for period @begin - @end',
      [
        '@begin' => $config['start_date'],
        '@end' => $config['end_date'],
      ]));
      $sheet->setCellValue("A3", t('Dividend'));
      $sheet->setCellValue("B3", $config['dividend']);
      $sheet->setCellValue("C3", $context['results']['base']);
      $sheet->setCellValue("A4", t('Total'));

      $sheet->setCellValue("A6", t('Shareholder Number'));
      $sheet->setCellValue("B6", t('Shareholder Name'));
      $sheet->setCellValue("C6", t('Shares (fraction)'));
      $sheet->setCellValue("D6", t('Tax Rate'));
      $sheet->setCellValue("E6", t('Gross'));
      $sheet->setCellValue("F6", t('Tax Amount'));
      $sheet->setCellValue("G6", t('Dividend'));
      $sheet->setCellValue("H6", t('Tax code'));

      $context['sandbox']['row'] = 7;
      $context['sandbox']['to_write'] = $context['results']['totals'];
    }
    else {
      $sheet = $spreadsheet->getActiveSheet();
    }

    for ($i = 0; $i < 5000; $i++) {
      if ($group = array_pop($context['sandbox']['to_write'])) {
        foreach ($group['grouped'] as $line) {
          $row = $context['sandbox']['row'];

          $shareholder = Shareholder::load($group['shareholder_id']);

          $sheet->setCellValue("A{$row}", $shareholder->getNumber());
          $sheet->setCellValue("B{$row}", $shareholder->getName());
          $sheet->setCellValue("C{$row}", $line['fraction']);
          $sheet->setCellValue("D{$row}", $line['tax']);
          $sheet->setCellValue("E{$row}", "=ROUND(C{$row}*B3/C3, 2)");
          $sheet->setCellValue("F{$row}", "=ROUND(E{$row}*(D{$row}/100), 2)");
          $sheet->setCellValue("G{$row}", "=E{$row}-F{$row}");
          $sheet->setCellValue("H{$row}", implode(", ", $line['taxcode']));

          $sheet->setCellValue("B4", "=SUM(E7:E{$row})");

          $context['sandbox']['row']++;
        }
      }
    }

    $writer = IOFactory::createWriter($spreadsheet, 'Xlsx');
    $writer->setPreCalculateFormulas(FALSE);
    $writer->save(drupal_realpath($context['results']['output']->getFileUri()));
    $context['results']['output']->save();
    $context['finished'] = (count($context['results']['totals']) - count($context['sandbox']['to_write'])) / count($context['results']['totals']);
  }

  /**
   * {@inheritdoc}
   */
  public static function finishExportDividend($success, $results, $operations) {
    drupal_set_message(t(
        'Exported @number dividend lines download: @url',
        [
          '@number' => count($results['details']),
          '@url' => Link::fromTextAndUrl('dividend.xlsx', Url::fromUri(file_create_url($results['output']->getFileUri())))->toString(),
        ]
    ));
  }


  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dividend_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $allocation_manager = \Drupal::service('plugin.manager.dividend_allocation_definitions');
    $tax_manager = \Drupal::service('plugin.manager.dividend_tax_definitions');

    $allocation_options = array_column($allocation_manager->getDefinitions(), 'label', 'id');
    $tax_options = array_column($tax_manager->getDefinitions(), 'label', 'id');

    $previous_year = date('Y') - 1;
    $this_year = date('Y');

    $form['dividend_date'] = [
      '#type' => 'date',
      '#title' => $this->t('Date of dividend'),
      '#description' => $this->t('Date of the dividend'),
      '#weight' => '0',
      '#default_value' => date('Y-m-d'),
    ];
    $form['begin_of_period'] = [
      '#type' => 'date',
      '#title' => $this->t('Begin of period'),
      '#description' => $this->t('Start date of the period'),
      '#weight' => '0',
      '#default_value' => "{$previous_year}-01-01",
    ];
    $form['end_of_period'] = [
      '#type' => 'date',
      '#title' => $this->t('End of period'),
      '#description' => $this->t('End date of the period'),
      '#weight' => '0',
      '#default_value' => "{$this_year}-01-01",
    ];
    $form['use_date'] = [
      '#type' => 'select',
      '#title' => $this->t('Date to use'),
      '#options' => [
        0 => $this->t('Validation date'),
        1 => $this->t('Payment date'),
      ],
      '#default_value' => 1,
      '#weight' => '0',
    ];
    $form['dividend'] = [
      '#type' => 'number',
      '#title' => $this->t('Dividend'),
      '#description' => $this->t('The dividend per share'),
      '#weight' => '0',
      '#default_value' => 15,
      '#step' => 0.0001,
    ];
    $form['allocation_method'] = [
      '#type' => 'select',
      '#title' => $this->t('Allocation method'),
      '#options' => $allocation_options,
      '#weight' => '0',
    ];
    $form['tax_method'] = [
      '#type' => 'select',
      '#title' => $this->t('Tax'),
      '#options' => $tax_options,
      '#weight' => '0',
    ];

    $form['action'] = [
      '#type' => 'radios',
      '#default_value' => 1,
      '#options' => [
        '1' => $this->t('Download dividend simulation'),
        '2' => $this->t('Register dividend'),
      ],
    ];

    $form['distribution'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Distribution name'),
      '#maxlength' => 50,
      '#states' => [
        'visible' => [
          ':input[name="action"]' => array('value' => '2'),
        ],
        'required' => [
          ':input[name="action"]' => array('value' => '2'),
        ],
      ],
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $begin = new \DateTime($form_state->getValue('begin_of_period'));
    $end = new \DateTime($form_state->getValue('end_of_period'));
    if ($begin >= $end) {
      $form_state->setErrorByName('begin_of_period', t('End of period should be after Begin of period!'));
    }
    if (!\Drupal::service('file_system')->validScheme('private')) {
      $form_state->setErrorByName('dividend', t('The private file system has to be set up to use this function!'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $s = \Drupal::service('shareholder_register_dividend.default');

    $dividend_config = [
      'dividend_date' => $form_state->getValue('dividend_date'),
      'dividend' => $form_state->getValue('dividend'),
      'start_date' => $form_state->getValue('begin_of_period'),
      'end_date' => $form_state->getValue('end_of_period'),
      'allocation' => $form_state->getValue('allocation_method'),
      'use_payment_date' => $form_state->getValue('use_date'),
      'tax' => $form_state->getValue('tax_method'),
      'total_hash' => "\Drupal\shareholder_register_dividend\DividendService::dividendTotalHash",
    ];

    if ($form_state->getValue('action') == '1') {
      $dividend_config['total_hash'] = "\Drupal\shareholder_register_dividend\DividendService::dividendTotalHashWithTaxDetails";
    }

    $batch = $s->getDividendAllocationBatch($dividend_config);

    if ($form_state->getValue('action') == '1') {
      $batch['operations'][] = [
        '\Drupal\shareholder_register_dividend\Form\DividendForm::writeExportDividend',
        [],
      ];
      $batch['operations'][] = [
        '\Drupal\shareholder_register_dividend\Form\DividendForm::writeExportDividendDetails',
        [],
      ];
      $batch['finished'] = '\Drupal\shareholder_register_dividend\Form\DividendForm::finishExportDividend';
    }
    else {
      $batch['operations'][] = [
        '\Drupal\shareholder_register_dividend\Form\DividendForm::registerDividend',
        [
          $form_state->getValue('distribution'),
        ],
      ];
      $batch['finished'] = '\Drupal\shareholder_register_dividend\Form\DividendForm::finishRegisterDividend';
    }

    batch_set($batch);
  }

}
