<?php

namespace Drupal\shareholder_register_dividend\Form;

use Symfony\Component\DependencyInjection\ContainerInterface;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;

use Drupal\shareholder_register\Entity\Shareholder;
use Drupal\shareholder_register_dividend\Entity\DividendDistribution;
use Drupal\shareholder_register_dividend\Entity\ShareholderDividend;
use Drupal\shareholder_register_dividend\DividendServiceInterface;

/**
 * Class ImportDividendForm.
 */
class ImportDividendForm extends FormBase {

  /**
   * @var DividendServiceInterface $account
   */
  protected $account;

  /**
   * Class constructor.
   */
  public function __construct(DividendServiceInterface $account) {
    $this->account = $account;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('shareholder_register_dividend.default')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function importDividend($shareholder_number, $gross, $net, $distribution, &$context) {
    $shareholder = Shareholder::loadByNumber($shareholder_number);
    if (!$shareholder) {
      return;
    }

    $dividend = ShareholderDividend::create([
      'name' => substr("{$shareholder->getName()} {$distribution->getName()}", 0, 50),
      'shareholder_id' => $shareholder,
      'distribution_id' => $distribution->id(),
      'gross' => $gross,
      'net' => $net,
    ]);
    $dividend->save();

    if (!isset($context['results']['count'])) {
      $context['results']['count'] = 0;
    }

    $context['results']['count'] += 1;
  }

  /**
   * {@inheritdoc}
   */
  public static function finishImportDividend($success, $results, $operations) {
    drupal_set_message(t(
        'Imported @number dividend lines from CSV File.',
        [
          '@number' => $results['count'],
        ]
    ));
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'import_dividend_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $columns = [];
    foreach (range(0, 25) as $c) {
      $columns[$c] = chr(ord('A') + $c);
    }

    $form['distribution'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Dividend distribution'),
      '#required' => TRUE,
    ];

    $form['dividend_csv_file'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('Dividend CSV file'),
      '#description' => $this->t('CSV file containing the dividends to be registered.'),
      '#weight' => '0',
      '#required' => TRUE,
      '#upload_validators' => [
        'file_validate_extensions' => ['csv']
      ],
    ];
    $form['shareholder_number_column'] = [
      '#type' => 'select',
      '#title' => $this->t('Shareholder Number Column'),
      '#description' => $this->t('Column in CSV which holds the shareholder number.'),
      '#options' => $columns,
      '#required' => TRUE,
      '#default_value' => 0,
    ];
    $form['gross_amount_column'] = [
      '#type' => 'select',
      '#title' => $this->t('Gross Amount Column'),
      '#description' => $this->t('Column in CSV which holds the gross amount.'),
      '#options' => $columns,
      '#required' => TRUE,
      '#default_value' => 4,
    ];
    $form['net_amount_column'] = [
      '#type' => 'select',
      '#title' => $this->t('Net Amount Column'),
      '#description' => $this->t('Column in CSV which holds the net amount.'),
      '#options' => $columns,
      '#required' => TRUE,
      '#default_value' => 6,
    ];
    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $batch = array(
      'title' => t('Importing'),
      'operations' => [],
      'finished' => '\Drupal\shareholder_register_dividend\Form\ImportDividendForm::finishImportDividend',
    );

    $distribution = DividendDistribution::create([
                      'name' => $form_state->getValue('distribution'),
    ]);
    $distribution->save();

    $csv_file = file_load(reset($form_state->getValue('dividend_csv_file')));
    if (($handle = fopen(drupal_realpath($csv_file->GetFileUri()), "r")) !== FALSE) {
      while (($data = fgetcsv($handle, 1000, ",")) !== FALSE) {
      /*
        if (!is_numeric($data[$form_state->getValue('shareholder_number_column')])) {
          continue;
        }
	*/
        if (!is_numeric($data[$form_state->getValue('gross_amount_column')])) {
          continue;
        }
        if (!is_numeric($data[$form_state->getValue('net_amount_column')])) {
          continue;
        }

        $batch['operations'][] = ['\Drupal\shareholder_register_dividend\Form\ImportDividendForm::importDividend', [
          $data[$form_state->getValue('shareholder_number_column')],
          $data[$form_state->getValue('gross_amount_column')],
          $data[$form_state->getValue('net_amount_column')],
          $distribution,
        ]];
      }
      fclose($handle);
    }
    batch_set($batch);
  }

}
