<?php
namespace Drupal\shareholder_register_dividend\Plugin\DividendAllocation;

use Drupal\shareholder_register_dividend\Plugin\DividendAllocationDefinitionBase;
use Drupal\shareholder_register_dividend\Exception\ShareholderRegisterDividendException;

/**
 * Prorata.
 *
 * @DividendAllocationDefinition(
 *   id = "prorata",
 *   label = @Translation("Prorata"),
 *  )
 */
class ProRata extends DividendAllocationDefinitionBase {

  /**
   * {@inheritdoc}
   */
  public function getDividendBase($dividend_config) {
    $interval = date_diff(
      date_create($dividend_config['start_date']),
      date_create($dividend_config['end_date']));
    return $interval->format('%a');
  }

  /**
   * {@inheritdoc}
   */
  public function computeDividend($shareholder, $dividend_config) {
    $result = [];

    $start_date = $dividend_config['start_date'];
    $end_date = $dividend_config['end_date'];
    $use_payment_date = isset($dividend_config['use_payment_date']) ? $dividend_config['use_payment_date'] : FALSE;

    $share_ids = $shareholder->getShareIdsAtDate($start_date, NULL, $use_payment_date);
    $current_shares = [];
    $current_date = $start_date;
    foreach ($share_ids as $share_id) {
      $current_shares[$share_id] = [
        'start_date' => $start_date,
      ];
    }

    foreach ($shareholder->getValidTransactionsInRange($start_date, $end_date, $use_payment_date) as $transaction) {
      if ($transaction->getQuantity() > 0) {
        foreach ($transaction->getShares() as $share) {
          $current_shares[$share->id()] = [
            'start_date' => $transaction->getTransactionDate($use_payment_date),
          ];
        }
      }
      else {
        foreach ($transaction->getShares() as $share) {
          if (!array_key_exists($share->id(), $current_shares)) {
            if ($transaction->getTransactionDate($use_payment_date) == $start_date) {
              continue;
            }
            throw new ShareholderRegisterDividendException(
              "Internal error: Share not found in previous transactions!");
          }

          $dividend = new \stdClass();
          $dividend->share_id = $share->id();
          $dividend->shareholder_id = $shareholder->id();
          $dividend->start_date = $current_shares[$share->id()]['start_date'];
          $dividend->end_date = $transaction->getTransactionDate($use_payment_date);
          $dividend->tax = 0;
          $dividend->fraction = date_diff(
            date_create($current_shares[$share->id()]['start_date']),
            date_create($transaction->getTransactionDate($use_payment_date)))->format('%a');
          $result[] = $dividend;
          unset($current_shares[$share->id()]);
        }
      }
    }

    foreach ($current_shares as $share_id => $share_data) {
      $dividend = new \stdClass();
      $dividend->share_id = $share_id;
      $dividend->shareholder_id = $shareholder->id();
      $dividend->start_date = $share_data['start_date'];
      $dividend->end_date = isset($share_data['end_date']) ? $share_data['end_date'] : $end_date;
      $dividend->tax = 0;
      $dividend->fraction = date_diff(
        date_create($share_data['start_date']),
        date_create(isset($share_data['end_date']) ? $share_data['end_date'] : $end_date))->format('%a');
      $result[] = $dividend;
    }
    return $result;
  }

}
