<?php
namespace Drupal\shareholder_register_dividend\Plugin\DividendAllocation;

use Drupal\shareholder_register_dividend\Plugin\DividendAllocationDefinitionBase;
use Drupal\shareholder_register_dividend\Exception\ShareholderRegisterDividendException;

/**
 * Prorata.
 *
 * @DividendAllocationDefinition(
 *   id = "prorata_month",
 *   label = @Translation("Prorata Monthly"),
 *  )
 */
class ProRataMonth extends DividendAllocationDefinitionBase {

  /**
   * The date at which to determine initial state.
   */
  protected static function getInitialDate($date) {
    return $date;
  }

  /**
   * Return the next first day of month.
   *
   * If date is the first of a month, return it, otherwise return the
   * first of next month.
   */
  protected static function startOfPeriod($date) {
    $d = new \DateTime($date);
    if ($d->format('d') == 1) {
      return $date;
    }
    else {
      return date('Y-m-d', strtotime("{$date} first day of next month"));
    }
  }

  /**
   * Return the previous first day of month.
   *
   * If date is the first of a month, return it, otherwise return the
   * first of this month.
   */
  protected static function endOfPeriod($date) {
    $d = new \DateTime($date);
    if ($d->format('d') == 1) {
      return $date;
    }
    else {
      return date('Y-m-d', strtotime("{$date} first day of this month"));
    }
  }

  /**
   * Date diff in months (years * 12 + months).
   */
  public static function diffTotalMonths($start_date, $end_date) {
    // Set timezone to UTC to avoid dst confusion
    // Ex: date_diff(date_create('2017-01-01'), date_create('2017-10-01')) -> interval: + 8m 30d.
    $backup_tz = date_default_timezone_get();
    date_default_timezone_set('UTC');

    $diff = date_diff(
      date_create($start_date),
      date_create($end_date));

    date_default_timezone_set($backup_tz);
    return $diff->format('%y') * 12 + $diff->format('%m');
  }

  /**
   * {@inheritdoc}
   */
  public function getDividendBase($dividend_config) {
    return static::diffTotalMonths(
      $dividend_config['start_date'],
      $dividend_config['end_date']);
  }

  /**
   * {@inheritdoc}
   */
  public function computeDividend($shareholder, $dividend_config) {
    $result = [];

    $start_date = $dividend_config['start_date'];
    $end_date = $dividend_config['end_date'];
    $use_payment_date = isset($dividend_config['use_payment_date']) ? $dividend_config['use_payment_date'] : FALSE;

    $shares = $shareholder->getSharesAtDate(static::getInitialDate($start_date), NULL, $use_payment_date);
    $current_shares = [];
    $current_date = $start_date;
    foreach ($shares as $share) {
      $current_shares[$share->id()] = [
        'share' => $share,
        'start_date' => static::startOfPeriod(
          static::getInitialDate($start_date)),
      ];
    }

    foreach ($shareholder->getValidTransactionsInRange($start_date, $end_date, $use_payment_date) as $transaction) {
      if ($transaction->getQuantity() > 0) {
        foreach ($transaction->getShares() as $share) {
          $current_shares[$share->id()] = [
            'share' => $share,
            'start_date' => static::startOfPeriod($transaction->getTransactionDate($use_payment_date)),
          ];
        }
      }
      else {
        foreach ($transaction->getShares() as $share) {
          if (!array_key_exists($share->id(), $current_shares)) {
            if ($transaction->getTransactionDate($use_payment_date) == $start_date) {
              continue;
            }
            throw new ShareholderRegisterDividendException(
              "Internal error: Share not found in previous transactions!");
          }

          $dividend = new \stdClass();
          $dividend->share_id = $share->id();
          $dividend->shareholder_id = $shareholder->id();
          $dividend->start_date = $current_shares[$share->id()]['start_date'];
          $dividend->end_date = $transaction->getTransactionDate($use_payment_date);
          $dividend->tax = 0;
          $dividend->fraction = static::diffTotalMonths(
            $current_shares[$share->id()]['start_date'],
            static::endOfPeriod($transaction->getTransactionDate($use_payment_date)));
          $result[] = $dividend;
          unset($current_shares[$share->id()]);
        }
      }
    }

    foreach ($current_shares as $share_data) {
      $dividend = new \stdClass();
      $dividend->share_id = $share_data['share']->id();
      $dividend->shareholder_id = $shareholder->id();
      $dividend->start_date = $share_data['start_date'];
      $dividend->end_date = isset($share_data['end_date']) ? $share_data['end_date'] : static::endOfPeriod($end_date);
      $dividend->tax = 0;
      $dividend->fraction = static::diffTotalMonths(
        $share_data['start_date'],
        static::endOfPeriod(isset($share_data['end_date']) ? $share_data['end_date'] : $end_date));
      $result[] = $dividend;
    }
    return $result;
  }

}
