<?php
namespace Drupal\shareholder_register_dividend\Plugin\DividendTax;

use Drupal\shareholder_register_dividend\Plugin\DividendTaxDefinitionBase;
use Drupal\shareholder_register\Entity\Share;

/**
 * Belgian 2018 Dividend RV tax.
 *
 * @DividendTaxDefinition(
 *   id = "BE2018VVPR",
 *   label = @Translation("BE 2018 (VVPR)"),
 *  )
 */
class BE2018VVPR extends DividendTaxDefinitionBase {

  /**
   * {@inheritdoc}
   */
  public function computeDividendTaxRate($shareholder, $dividends, $dividend_config) {
    $use_payment_date = TRUE;

    // Totale kapitaalverlaging sinds 01/05/2013.
    $transactions = $shareholder->getValidTransactionsInRange(
      '2013-05-01',
      date('Y-m-d'),
      $use_payment_date
    );
    $min_value = array_reduce(
      array_filter(
        $transactions,
        function ($t) {
          return $t->getQuantity < 0;
        }),
      function ($result, $transaction) {
        return bcadd(
          $result,
          bcmul(
            $transaction->getQuantity(),
            bcadd(
              $transaction->get('share_value')->value,
              $transaction->get('issue_premium')->value,
              2),
            2),
          2);
      },
      0);
    $plus_value = 0;
    foreach ($dividends as $dividend) {
      $share = Share::load($dividend->share_id);

      $issue_date = $share->getIssuingTransaction()->getTransactionDate($use_payment_date);
      if ($issue_date < '2013-07-01') {
        $dividend->tax = 30;
        $dividend->taxcode = 'vvpr-pre2013';
        continue;
      }
      if ($share->getShareholder()->bundle() != 'natural_person') {
        $dividend->tax = 30;
        $dividend->taxcode = 'vvpr-na';
        continue;
      }

      if ($min_value < $plus_value) {
        $plus_value = bcadd(
          $plus_value,
          bcadd(
            $share->get('share_value')->value,
            $share->get('issue_premium')->value,
            2),
          2);
        $dividend->tax = 30;
        $dividend->taxcode = 'vvpr-abuse';
        continue;
      }

      $issue_year = substr($issue_date, 0, 4);
      $dividend_year = substr($dividend_config['start_date'], 0, 4);
      if (($dividend_year - $issue_year) > 2) {
        $dividend->tax = 15;
        $dividend->taxcode = 'vvpr-year-2';
      }
      elseif (($dividend_year - $issue_year) > 1) {
        $dividend->tax = 20;
        $dividend->taxcode = 'vvpr-year-1';
      }
      else {
        $dividend->tax = 30;
        $dividend->taxcode = 'vvpr-year-0';
      }
    }
  }

}
