<?php

namespace Drupal\shareholder_register_dividend\Plugin\Field\FieldFormatter;

use Symfony\Component\DependencyInjection\ContainerInterface;

use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;

use Drupal\shareholder_register_dividend\Entity\ShareholderDividend;
use Drupal\shareholder_register\ShareholderRegisterFormatterServiceInterface;

/**
 * Plugin implementation of the 'shareholder_dividends_formatter' formatter.
 *
 * @FieldFormatter(
 *   id = "distribution_dividends_formatter",
 *   label = @Translation("Distribution Dividends formatter"),
 *   field_types = {
 *     "shareholder_dividends"
 *   }
 * )
 */
class DistributionDividendsFormatter extends FormatterBase {

  /**
   * ShareholderRegisterFormatterServiceInterface definition.
   *
   * @var \Drupal\shareholder_register\ShareholderRegisterFormatterServiceInterface
   */
  protected $shareholderRegisterFormatter;

  /**
   * Constructs a ShareholderDividendsFormatter instance.
   *
   * @param string $plugin_id
   *   The plugin_id for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings settings.
   * @param \Drupal\shareholder_register\ShareholderRegisterFormatterServiceInterface $formatterService
   *   The Shareholder Register Formatter service.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, $label, $view_mode, array $third_party_settings, ShareholderRegisterFormatterServiceInterface $formatterService) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->shareholderRegisterFormatter = $formatterService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('shareholder_register.formatter')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      // Implement default settings.
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    return [
      // Implement settings form.
    ] + parent::settingsForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    // Implement settings summary.

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements[0] = [
      '#type' => 'table',
      '#cache' => [
        'tags' =>
        \Drupal::entityTypeManager()->getDefinition('shareholder_dividend')->getListCacheTags(),
      ],
      '#header' => [
        t('Shareholder'),
        t('Gross'),
        t('Net'),
      ],
    ];

    foreach ($items as $delta => $item) {
      $elements[0]['#rows'][$delta] = $this->viewValue($item);
    }

    return $elements;
  }

  /**
   * Generate the output appropriate for one field item.
   *
   * @param \Drupal\Core\Field\FieldItemInterface $item
   *   One field item.
   *
   * @return string
   *   The textual output generated.
   */
  protected function viewValue(FieldItemInterface $item) {
    $value = $item->getValue();
    if (isset($value['entity'])) {
      $dividend = $value['entity'];
    }
    else {
      $dividend = ShareholderDividend::load($value['target_id']);
    }

    return [
      'shareholder' => $dividend->getShareholder()->label(),
      'gross' => $dividend->getGross(),
      'net' => $dividend->getNet(),
    ];
  }

}
