<?php
namespace Drupal\shareholder_register_dividend\TwigExtension;

use Drupal\shareholder_register\Entity\Share;

use Drupal\Core\StringTranslation\StringTranslationTrait;

use Drupal\shareholder_register\ShareholderRegisterFormatterServiceInterface;
use Drupal\shareholder_register_dividend\DividendServiceInterface;

/**
 * Class ShareholderRegisterDividendTwigExtension.
 */
class ShareholderRegisterDividendTwigExtension extends \Twig_Extension {

  use StringTranslationTrait;

  /**
   * Drupal\shareholder_register_dividend\DividendServiceInterface definition.
   *
   * @var \Drupal\shareholder_register_dividend\DividendServiceInterface
   */
  protected $dividendService;

  /**
   * Drupal\shareholder_register\ShareholderRegisterFormatterServiceInterface definition.
   *
   * @var \Drupal\shareholder_register\ShareholderRegisterFormatterServiceInterface
   */
  protected $shareholderRegisterFormatter;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    DividendServiceInterface $dividendService,
    ShareholderRegisterFormatterServiceInterface $shareholderRegisterFormatter) {

    $this->dividendService = $dividendService;
    $this->shareholderRegisterFormatter = $shareholderRegisterFormatter;
  }

  /**
   * {@inheritdoc}
   */
  public function getTokenParsers() {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getNodeVisitors() {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getFilters() {
    return [
      new \Twig_SimpleFilter('shareholder_dividend_gross', [$this, 'shareholderDividendGross']),
      new \Twig_SimpleFilter('shareholder_dividend_net', [$this, 'shareholderDividendNet']),
      new \Twig_SimpleFilter('shareholder_dividend_tax', [$this, 'shareholderDividendTax']),
      new \Twig_SimpleFilter('shareholder_dividend_details', [$this, 'shareholderDividendDetails']),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getTests() {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getFunctions() {
    return [
      new \Twig_SimpleFunction('shareholder_valid_dividends', [$this, 'shareholderValidDividends']),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getOperators() {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getName() {
    return 'shareholder_register_dividend.twig.extension';
  }

  /**
   * {@inheritdoc}
   */
  public function shareholderDividendGross($object, $distribution_id) {
    $c = $this->dividendService->getDividendContext($object, $distribution_id);
    return $c['gross'];
  }

  /**
   * {@inheritdoc}
   */
  public function shareholderDividendNet($object, $distribution_id) {
    $c = $this->dividendService->getDividendContext($object, $distribution_id);
    return $c['net'];
  }

  /**
   * {@inheritdoc}
   */
  public function shareholderDividendTax($object, $distribution_id) {
    $c = $this->dividendService->getDividendContext($object, $distribution_id);
    return $c['tax'];
  }

  /**
   * {@inheritdoc}
   */
  public function shareholderDividendDetails($object, $distribution_id) {
    $c = $this->dividendService->getDividendContext($object, $distribution_id);

    $result = [
      '#type' => 'table',
      '#header' => [
        $this->t('Shares and periods'),
        $this->t('Tax %'),
        $this->t('Gross'),
        $this->t('Tax'),
        $this->t('Net'),
      ],
      '#rows' => [],
      '#attributes' => [
        'class' => ['dividend-details'],
      ],
    ];

    foreach ($c['dividends'] as $dividend) {
      $details = $dividend->getData('details');
      foreach ($details['grouped'] as $tax_rate => $sub_dividend) {

        $groups = [];
        foreach ($sub_dividend['details'] as $detail) {
          $date_hash = "{$detail[start_date]}-{$detail[end_date]}";
          if (!isset($groups[$date_hash])) {
            $groups[$date_hash] = [
              'start_date' => $detail['start_date'],
              'end_date' => $detail['end_date'],
              'share_ids' => [],
            ];
          }
          $groups[$date_hash]['share_ids'][] = $detail['share_id'];
        }

        $share_details = [];
        foreach ($groups as $group) {
          $shares = Share::loadMultiple($group['share_ids']);
          $share_names = $this->shareholderRegisterFormatter->sharesToRanges($shares);
          $start_date_formatted = $this->shareholderRegisterFormatter->formatDate(
            $group['start_date'], 'shareholder_register_date');
          $end_date_formatted = $this->shareholderRegisterFormatter->formatDate(
            $this->shareholderRegisterFormatter->endDateIncl($group['end_date']), 'shareholder_register_date');
          $share_details[] = "{$start_date_formatted} - {$end_date_formatted}: {$share_names}";
        }

        $result['#rows'][] = [
          implode('<br/>', $share_details),
          "{$tax_rate} (" . implode(', ', $sub_dividend[taxcode]) . ")",
          $sub_dividend['gross'],
          $sub_dividend['tax'],
          $sub_dividend['net'],
        ];
      }
    }

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function shareholderValidDividends($object) {
    return array_filter(
      $object->get('dividends')->referencedEntities(),
      function ($d) {
        return $d->getDistribution()->getState() == 'valid';
      }
    );
  }

}
