<?php

namespace Drupal\Tests\shareholder_register_dividend\Kernel;

use Drupal\shareholder_register\Entity\Shareholder;
use Drupal\shareholder_register_dividend\DividendService;

use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\shareholder_register\Traits\LoadRegisterTrait;

/**
 * Tests Dividend.
 *
 * @group shareholder_register
 */
class DividendTest extends KernelTestBase {

  use LoadRegisterTrait;

  /**
   * Modules to enable.
   *
   * @var array
   */
  public static $modules = [
    'system',
    'user',
    'field',
    'file',
    'text',
    'datetime',
    'mail_template',
    'workflows',
    'simple_workflows',
    'shareholder_register',
    'shareholder_register_dividend',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp() {
    parent::setUp();
    $this->traitSetUp();
    $this->installEntitySchema('dividend_distribution');
    $this->installEntitySchema('shareholder_dividend');
    $this->installConfig('shareholder_register');
    $this->installConfig('shareholder_register_dividend');
  }

  /**
   * Test Prorata allocation.
   */
  public function testProrataAllocation() {
    $register = $this->loadBasicRegister(
      [
        'shareholders' => [
          [
            'number' => 1,
          ],
        ],
        'transactions' => [
          [
            'quantity' => 5,
            'shareholder_number' => 1,
            'payment_date' => '2017-01-02',
            'validation_date' => '2017-02-02',
          ],
        ],
      ]);

    $dividend_config = [
      'dividend_date' => '2018-03-01',
      'dividend' => 15,
      'start_date' => '2017-01-01',
      'end_date' => '2018-01-01',
      'allocation' => 'prorata',
      'use_payment_date' => 0,
      'tax' => 'BE2018',
      'total_hash' => "\Drupal\shareholder_register_dividend\DividendService::dividendTotalHash",
    ];

    $shareholder = Shareholder::load(reset($register['shareholder']));
    $plugin = DividendService::getDividendAllocationDefinition($dividend_config['allocation']);
    $result = $plugin->computeDividend($shareholder, $dividend_config);

    $this->assertEqual(count($result), 5);
    $this->assertEqual($result[0]->fraction, 333);
    $this->assertEqual($result[0]->start_date, '2017-02-02');
    $this->assertEqual($result[0]->end_date, '2018-01-01');

    // Test payment date computation.
    $dividend_config['use_payment_date'] = 1;

    $plugin = DividendService::getDividendAllocationDefinition($dividend_config['allocation']);
    $result = $plugin->computeDividend($shareholder, $dividend_config);

    $this->assertEqual(count($result), 5);
    $this->assertEqual($result[0]->fraction, 364);
    $this->assertEqual($result[0]->start_date, '2017-01-02');
    $this->assertEqual($result[0]->end_date, '2018-01-01');
  }

  /**
   * Test field_no_dividend.
   */
  public function testFieldNoDividend() {
    $register = $this->loadBasicRegister(
      [
        'shareholders' => [
          [
            'number' => 1,
            'field_no_dividend' => 1,
          ],
          [
            'number' => 2,
          ],
        ],
        'transactions' => [
          [
            'quantity' => 5,
            'shareholder_number' => 1,
            'payment_date' => '2017-01-02',
            'validation_date' => '2017-02-02',
          ],
          [
            'quantity' => 5,
            'shareholder_number' => 2,
            'payment_date' => '2017-01-02',
            'validation_date' => '2017-02-02',
          ],
        ],
      ]);

    $dividend_config = [
      'dividend_date' => '2018-03-01',
      'dividend' => 15,
      'start_date' => '2017-01-01',
      'end_date' => '2018-01-01',
      'allocation' => 'prorata',
      'use_payment_date' => 0,
      'tax' => 'BE2018',
      'total_hash' => "\Drupal\shareholder_register_dividend\DividendService::dividendTotalHash",
    ];

    $context = [];
    DividendService::initDividendBatch($dividend_config, $context);
    DividendService::dividendBatch($dividend_config, $context);

    $this->assertEqual(count($context['results']['totals']), 1);
  }

  /**
   * Test Prorata Month allocation 2.
   */
  public function testDividendTotal() {
    $register = $this->loadBasicRegister(
      [
        'shareholders' => [
          [
            'number' => 1,
          ],
        ],
        'transactions' => [
          [
            'quantity' => 5,
            'shareholder_number' => 1,
            'payment_date' => '2017-02-02',
            'validation_date' => '2017-02-02',
          ],
          [
            'quantity' => -5,
            'shareholder_number' => 1,
            'validation_date' => '2017-12-30',
          ],
        ],
      ]);

    $dividend_config = [
      'dividend_date' => '2018-03-01',
      'dividend' => 15,
      'start_date' => '2017-01-01',
      'end_date' => '2018-01-01',
      'allocation' => 'prorata',
      'use_payment_date' => 0,
      'tax' => 'BE2018',
      'total_hash' => "\Drupal\shareholder_register_dividend\DividendService::dividendTotalHash",
    ];

    $shareholder = Shareholder::load(reset($register['shareholder']));
    $plugin = DividendService::getDividendAllocationDefinition($dividend_config['allocation']);
    $result = $plugin->computeDividend($shareholder, $dividend_config);

    $this->assertEqual(count($result), 5);
    $this->assertEqual($result[0]->fraction, 331);
    $this->assertEqual($result[0]->start_date, '2017-02-02');
    $this->assertEqual($result[0]->end_date, '2017-12-30');

    $plugin = DividendService::getDividendTaxDefinition($dividend_config['tax']);
    $plugin->computeDividendTaxRate(
      $shareholder,
      $result,
      $dividend_config
    );

    $context = [];
    DividendService::initDividendBatch($dividend_config, $context);
    DividendService::dividendBatch($dividend_config, $context);

    $totals = $context['results']['totals'];

    $this->assertEqual(count($totals), 1);
    $this->assertEqual(reset($totals)['fraction'], 1655);
  }

}
