<?php

namespace Drupal\Tests\shareholder_register_dividend\Kernel;

use Drupal\shareholder_register\Entity\Shareholder;
use Drupal\shareholder_register_dividend\DividendService;

use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\shareholder_register\Traits\LoadRegisterTrait;

/**
 * Tests Dividend.
 *
 * @group shareholder_register
 */
class ProrataMonthTest extends KernelTestBase {

  use LoadRegisterTrait;

  /**
   * Modules to enable.
   *
   * @var array
   */
  public static $modules = [
    'system',
    'user',
    'file',
    'text',
    'datetime',
    'simple_workflows',
    'shareholder_register',
    'shareholder_register_dividend',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp() {
    parent::setUp();
    $this->traitSetUp();
    $this->installEntitySchema('dividend_distribution');
    $this->installEntitySchema('shareholder_dividend');
  }

  /**
   * Test previously owned share fraction.
   */
  public function testPreviouslyOwnedShareFraction() {
    $register = $this->loadBasicRegister(
      [
        'shareholders' => [
          [
            'number' => 1,
          ],
        ],
        'transactions' => [
          [
            'quantity' => 5,
            'shareholder_number' => 1,
            'validation_date' => '2012-02-01',
          ],
        ],
      ]);

    $dividend_config = [
      'dividend_date' => '2018-03-01',
      'dividend' => 15,
      'start_date' => '2017-01-01',
      'end_date' => '2018-01-01',
      'allocation' => 'prorata_month',
      'use_payment_date' => 0,
      'tax' => 'BE2018',
      'total_hash' => "\Drupal\shareholder_register_dividend\DividendService::dividendTotalHash",
    ];

    $shareholder = Shareholder::load(reset($register['shareholder']));
    $plugin = DividendService::getDividendAllocationDefinition($dividend_config['allocation']);
    $result = $plugin->computeDividend($shareholder, $dividend_config);

    $this->assertEqual(count($result), 5);
    $this->assertEqual($result[0]->fraction, 12);
    $this->assertEqual($result[0]->start_date, '2017-01-01');
    $this->assertEqual($result[0]->end_date, '2018-01-01');
  }

  /**
   * Test first day of month inclusive.
   */
  public function testFirstIncl() {
    $register = $this->loadBasicRegister(
      [
        'shareholders' => [
          [
            'number' => 1,
          ],
        ],
        'transactions' => [
          [
            'quantity' => 5,
            'shareholder_number' => 1,
            'validation_date' => '2017-02-01',
          ],
        ],
      ]);

    $dividend_config = [
      'dividend_date' => '2018-03-01',
      'dividend' => 15,
      'start_date' => '2017-01-01',
      'end_date' => '2018-01-01',
      'allocation' => 'prorata_month',
      'use_payment_date' => 0,
      'tax' => 'BE2018',
      'total_hash' => "\Drupal\shareholder_register_dividend\DividendService::dividendTotalHash",
    ];

    $shareholder = Shareholder::load(reset($register['shareholder']));
    $plugin = DividendService::getDividendAllocationDefinition($dividend_config['allocation']);
    $result = $plugin->computeDividend($shareholder, $dividend_config);

    $this->assertEqual(count($result), 5);
    $this->assertEqual($result[0]->fraction, 11);
    $this->assertEqual($result[0]->start_date, '2017-02-01');
    $this->assertEqual($result[0]->end_date, '2018-01-01');
  }

  /**
   * Test repurchased share fraction.
   */
  public function testRepurchasedShareFraction() {
    $register = $this->loadBasicRegister(
      [
        'shareholders' => [
          [
            'number' => 1,
          ],
        ],
        'transactions' => [
          [
            'quantity' => 5,
            'shareholder_number' => 1,
            'validation_date' => '2012-02-01',
          ],
          [
            'quantity' => -5,
            'shareholder_number' => 1,
            'validation_date' => '2017-10-05',
          ],
        ],
      ]);

    $dividend_config = [
      'dividend_date' => '2018-03-01',
      'dividend' => 15,
      'start_date' => '2017-01-01',
      'end_date' => '2018-01-01',
      'allocation' => 'prorata_month',
      'use_payment_date' => 0,
      'tax' => 'BE2018',
      'total_hash' => "\Drupal\shareholder_register_dividend\DividendService::dividendTotalHash",
    ];

    $shareholder = Shareholder::load(reset($register['shareholder']));
    $plugin = DividendService::getDividendAllocationDefinition($dividend_config['allocation']);
    $result = $plugin->computeDividend($shareholder, $dividend_config);

    $this->assertEqual(count($result), 5);
    $this->assertEqual($result[0]->fraction, 9);
    $this->assertEqual($result[0]->start_date, '2017-01-01');
    $this->assertEqual($result[0]->end_date, '2017-10-05');
  }

  /**
   * Test repurchased share fraction.
   */
  public function testRepurchasedFirstShareFraction() {
    $register = $this->loadBasicRegister(
      [
        'shareholders' => [
          [
            'number' => 1,
          ],
        ],
        'transactions' => [
          [
            'quantity' => 5,
            'shareholder_number' => 1,
            'validation_date' => '2012-02-01',
          ],
          [
            'quantity' => -5,
            'shareholder_number' => 1,
            'validation_date' => '2017-10-01',
          ],
        ],
      ]);

    $dividend_config = [
      'dividend_date' => '2018-03-01',
      'dividend' => 15,
      'start_date' => '2017-01-01',
      'end_date' => '2018-01-01',
      'allocation' => 'prorata_month',
      'use_payment_date' => 0,
      'tax' => 'BE2018',
      'total_hash' => "\Drupal\shareholder_register_dividend\DividendService::dividendTotalHash",
    ];

    $shareholder = Shareholder::load(reset($register['shareholder']));
    $plugin = DividendService::getDividendAllocationDefinition($dividend_config['allocation']);
    $result = $plugin->computeDividend($shareholder, $dividend_config);

    $this->assertEqual(count($result), 5);
    $this->assertEqual($result[0]->fraction, 9);
    $this->assertEqual($result[0]->start_date, '2017-01-01');
    $this->assertEqual($result[0]->end_date, '2017-10-01');
  }

  /**
   * Test Prorata Month allocation.
   */
  public function testProrataMonthAllocation() {
    $register = $this->loadBasicRegister(
      [
        'shareholders' => [
          [
            'number' => 1,
          ],
        ],
        'transactions' => [
          [
            'quantity' => 5,
            'shareholder_number' => 1,
            'payment_date' => '2017-01-02',
            'validation_date' => '2017-02-02',
          ],
        ],
      ]);

    $dividend_config = [
      'dividend_date' => '2018-03-01',
      'dividend' => 15,
      'start_date' => '2017-01-01',
      'end_date' => '2018-01-01',
      'allocation' => 'prorata_month',
      'use_payment_date' => 0,
      'tax' => 'BE2018',
      'total_hash' => "\Drupal\shareholder_register_dividend\DividendService::dividendTotalHash",
    ];

    $shareholder = Shareholder::load(reset($register['shareholder']));
    $plugin = DividendService::getDividendAllocationDefinition($dividend_config['allocation']);
    $result = $plugin->computeDividend($shareholder, $dividend_config);

    $this->assertEqual(count($result), 5);
    $this->assertEqual($result[0]->fraction, 10);
    $this->assertEqual($result[0]->start_date, '2017-03-01');
    $this->assertEqual($result[0]->end_date, '2018-01-01');

    // Test payment date computation.
    $dividend_config['use_payment_date'] = 1;

    $plugin = DividendService::getDividendAllocationDefinition($dividend_config['allocation']);
    $result = $plugin->computeDividend($shareholder, $dividend_config);

    $this->assertEqual(count($result), 5);
    $this->assertEqual($result[0]->fraction, 11);
    $this->assertEqual($result[0]->start_date, '2017-02-01');
    $this->assertEqual($result[0]->end_date, '2018-01-01');
  }

  /**
   * Test Prorata Month allocation 2.
   */
  public function testProrataMonth2Allocation() {
    $register = $this->loadBasicRegister(
      [
        'shareholders' => [
          [
            'number' => 1,
          ],
        ],
        'transactions' => [
          [
            'quantity' => 5,
            'shareholder_number' => 1,
            'payment_date' => '2017-02-02',
            'validation_date' => '2017-02-02',
          ],
          [
            'quantity' => -5,
            'shareholder_number' => 1,
            'validation_date' => '2017-12-30',
          ],
        ],
      ]);

    $dividend_config = [
      'dividend_date' => '2018-03-01',
      'dividend' => 15,
      'start_date' => '2017-01-01',
      'end_date' => '2018-01-01',
      'allocation' => 'prorata_month',
      'use_payment_date' => 0,
      'tax' => 'BE2018',
      'total_hash' => "\Drupal\shareholder_register_dividend\DividendService::dividendTotalHash",
    ];

    $shareholder = Shareholder::load(reset($register['shareholder']));
    $plugin = DividendService::getDividendAllocationDefinition($dividend_config['allocation']);
    $result = $plugin->computeDividend($shareholder, $dividend_config);

    $this->assertEqual(count($result), 5);
    $this->assertEqual($result[0]->fraction, 9);
    $this->assertEqual($result[0]->start_date, '2017-03-01');
    $this->assertEqual($result[0]->end_date, '2017-12-30');
  }

}
