<?php

namespace Drupal\shareholder_register_taxshelter\Entity;

use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityPublishedTrait;
use Drupal\Core\Entity\EntityTypeInterface;

/**
 * Defines the Taxshelter claim entity.
 *
 * @ingroup shareholder_register_taxshelter
 *
 * @ContentEntityType(
 *   id = "taxshelter_claim",
 *   label = @Translation("Taxshelter claim"),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\shareholder_register_taxshelter\TaxshelterClaimListBuilder",
 *     "views_data" = "Drupal\shareholder_register_taxshelter\Entity\TaxshelterClaimViewsData",
 *     "access" = "Drupal\shareholder_register_taxshelter\TaxshelterClaimAccessControlHandler",
 *     "route_provider" = {
 *       "html" = "Drupal\shareholder_register_taxshelter\TaxshelterClaimHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "taxshelter_claim",
 *   translatable = FALSE,
 *   admin_permission = "administer taxshelter claim entities",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "name",
 *     "uuid" = "uuid",
 *     "published" = "status",
 *   },
 *   links = {
 *     "canonical" = "/admin/shareholder_register/tools/taxshelter_claim/{taxshelter_claim}",
 *     "collection" = "/admin/shareholder_register/tools/taxshelter_claims",
 *   },
 *   field_ui_base_route = "taxshelter_claim.settings"
 * )
 */
class TaxshelterClaim extends ContentEntityBase implements TaxshelterClaimInterface {

  use EntityChangedTrait;
  use EntityPublishedTrait;

  /**
   * {@inheritdoc}
   */
  public function label() {
    return "{$this->id()}";
  }

  /**
   * {@inheritdoc}
   */
  public function getType() {
    return $this->get('type')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getYear() {
    return $this->get('year')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getShareholder() {
    return $this->get('shareholder_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getAmount() {
    return $this->get('amount')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getShares() {
    return $this->get('shares')->referencedEntities();
  }

  /**
   * {@inheritdoc}
   */
  public function getDate() {
    $shares = $this->get('shares')->referencedEntities();
    return reset($shares)->getPaymentDate();
  }

  /**
   * {@inheritdoc}
   */
  public function getBelcotaxSerial() {
    return $this->get('belcotax_serial')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getData($key, $default = NULL) {
    $data = [];
    if (!$this->get('extra')->isEmpty()) {
      $data = $this->get('extra')->first()->getValue();
    }
    return isset($data[$key]) ? $data[$key] : $default;
  }

  /**
   * {@inheritdoc}
   */
  public function setData($key, $value) {
    $this->get('extra')->__set($key, $value);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime($timestamp) {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    // Add the published field.
    $fields += static::publishedBaseFieldDefinitions($entity_type);

    $fields['type'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Type'))
      ->setDescription(t('The type of the Taxshelter claim entity.'))
      ->setSettings([
        'max_length' => 50,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    $fields['year'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Year'))
      ->setDescription(t('The year of the Taxshelter claim entity.'))
      ->setSettings([
        'max_length' => 50,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    $fields['belcotax_serial'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Belcotax Serial Number'))
      ->setDescription(t('The Belcotax Serial Number Taxshelter claim entity.'));

    $fields['shareholder_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Shareholder'))
      ->setDescription(t('The Shareholder.'))
      ->setSettings(array(
        'target_type' => 'shareholder',
        'default_value' => 0,
      ))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'weight' => 2,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['shares'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Shares'))
      ->setSettings([
        'target_type' => 'share',
      ])
      ->setCardinality(FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'weight' => 15,
      ]);

    $fields['amount'] = BaseFieldDefinition::create('decimal')
      ->setLabel(t('Amount'))
      ->setDescription(t('The invested amount.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'number_decimal',
        'weight' => 6,
      ])
      ->setDisplayOptions('form', [
        'type' => 'number',
        'weight' => 6,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['statement'] = BaseFieldDefinition::create('file')
      ->setLabel(t('Statement'))
      ->setDescription(t('The Statement.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'weight' => 2,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['extra'] = BaseFieldDefinition::create('map')
      ->setLabel(t('Extra'))
      ->setDescription(t('Extra data for this Taxshelter Claim.'));

    $fields['status']->setDescription(t('A boolean indicating whether the Taxshelter claim is published.'))
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'weight' => -3,
      ]);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the entity was last edited.'));

    return $fields;
  }

}
