<?php

namespace Drupal\shareholder_register_taxshelter\Form;

use Symfony\Component\DependencyInjection\ContainerInterface;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;

use Drupal\shareholder_register\ShareholderRegisterServiceInterface;
use Drupal\shareholder_register_taxshelter\ShareholderRegisterTaxshelterServiceInterface;

/**
 * Class ShareholderRegisterTaxshelter.
 */
class ShareholderRegisterTaxshelter extends FormBase {

  /**
   * @var \Drupal\shareholder_register_taxshelter\ShareholderRegisterTaxshelterServiceInterface
   */
  protected $taxshelterService;

  /**
   * @var \Drupal\shareholder_register\ShareholderRegisterServiceInterface
   */
  protected $shareholderRegisterService;

  /**
   * Class constructor.
   */
  public function __construct(
    ShareholderRegisterTaxshelterServiceInterface $taxshelterService,
    ShareholderRegisterServiceInterface $shareholderRegisterService
  ) {
    $this->taxshelterService = $taxshelterService;
    $this->shareholderRegisterService = $shareholderRegisterService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('shareholder_register_taxshelter.default'),
      $container->get('shareholder_register.default')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'shareholder_register_taxshelter_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $max_date = $this->taxshelterService->getLatestTaxshelterDate();

    $form['help'] = [
      '#markup' => t('Generate the taxshelter documentation for the given year.'),
      '#prefix' => "<p>",
      '#suffix' => "</p>",
    ];
    $form['warning'] = [
      '#markup' => t('You have to manually verify all these documents, as not all the conditions can be checked.'),
      '#prefix' => "<p>",
      '#suffix' => "</p>",
    ];
    $form['year'] = [
      '#type' => 'textfield',
      '#disabled' => TRUE,
      '#title' => $this->t("Year"),
      '#default_value' => date("Y", strtotime("-1 year")),
    ];
    $form['max_date'] = [
      '#type' => 'date',
      '#title' => $this->t("Latest date for taxshelter"),
      '#default_value' => $max_date,
      '#states' => [
        'enabled' => [
          ':input[id="edit-override-max-date"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['override_max_date'] = [
      '#type' => 'checkbox',
      '#title' => $this->t("Override max date for taxshelter."),
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit'),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $entity_ids = \Drupal::entityQuery('taxshelter_claim')
      ->condition('year', trim($form_state->getValue('year')))
      ->execute();
    if (count($entity_ids)) {
      $form_state->setErrorByName('year', $this->t('There are already taxshelter documents for the selected year!'));
    }

    foreach ($this->taxshelterService->checkTaxshelterConfig() as $msg) {
      $form_state->setErrorByName('year', $msg[1]);
    }
    foreach ($this->shareholderRegisterService->checkShareholderRegisterConfig() as $msg) {
      $form_state->setErrorByName('year', $msg[1]);
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $batch = $this->taxshelterService->getTaxshelterBatch(trim($form_state->getValue('year')), $form_state->getValue('max_date'));
    batch_set($batch);
  }

}
