<?php
namespace Drupal\shareholder_register_taxshelter\Plugin\Action;

use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

use Drupal\views_bulk_operations\Action\ViewsBulkOperationsActionBase;

/**
 * Mail Shareholder Extract.
 *
 * @Action(
 *   id = "mail_taxshelter_action",
 *   label = @Translation("Mail Taxshelter"),
 *   type = "shareholder"
 * )
 */
class MailTaxshelterAction extends ViewsBulkOperationsActionBase {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'year' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['year'] = [
      '#type' => 'textfield',
      '#disabled' => TRUE,
      '#title' => $this->t("Year"),
      '#default_value' => date("Y", strtotime("-1 year")),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    $invalid_taxshelter_ids = \Drupal::EntityQuery('taxshelter_claim')
      ->condition('year', $form_state->getValue('year'));
    $orGroup = $invalid_taxshelter_ids->orConditionGroup();
    $orGroup->condition('statement.target_id.entity.uri', '');
    $orGroup->condition('statement.target_id.entity.filesize', 0, '<=');
    $invalid_taxshelter_ids->condition($orGroup);

    if (count($invalid_taxshelter_ids->execute())) {
      $form_state->setErrorByName(
        'year',
        $this->t("Claims with invalid documents selected ':ids'",
        [
          ':ids' => implode(',', $invalid_taxshelter_ids->execute()),
        ])
      );
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute($entity = NULL) {
    if ($entity) {
      $taxshelter_ids = \Drupal::EntityQuery('taxshelter_claim')
        ->condition('shareholder_id', $entity->id())
        ->condition('year', $this->configuration['year'])
        ->execute();

      if (count($taxshelter_ids)) {
        \Drupal::service('shareholder_register_taxshelter.default')->mailTaxshelterForShareholder(
          $entity,
          $this->configuration['year']);
      }
      else {
        \Drupal::messenger()->addMessage($this->t(
            'No taxshelter documents for :name, so no mail was sent!',
            [
              ':name' => $entity->getName(),
            ])
        );
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, AccountInterface $account = NULL, $return_as_object = FALSE) {
    if ($object->getState() != 'valid') {
      $result = AccessResult::forbidden();
    }
    else {
      $result = $object->access('view', $account, TRUE);
    }
    return $return_as_object ? $result : $result->isAllowed();
  }

}
