<?php

namespace Drupal\shareholder_register_taxshelter;

use Drupal\Core\Link;
use Drupal\Core\Url;

use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

use Drupal\file\Entity\File;

use Drupal\shareholder_register_taxshelter\Entity\TaxshelterClaim;
use Drupal\shareholder_register_taxshelter\Entity\TaxshelterBelcotax;

use Drupal\shareholder_register\Entity\Share;
use Drupal\shareholder_register\Entity\ShareTransaction;

use Sabre\Xml\Service as XmlService;
use mikehaertl\wkhtmlto\Pdf;

/**
 * Class ShareholderRegisterTaxshelterService.
 */
class ShareholderRegisterTaxshelterService implements ShareholderRegisterTaxshelterServiceInterface {

  use StringTranslationTrait;

  const TAXSHELTER_LIMIT = 250000;
  const TAXSHELTER_SHAREHOLDER_LIMIT = 100000;

  /**
   * Drupal\Core\Messenger\MessengerInterface definition.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a new ShareholderRegisterTaxshelterService object.
   */
  public function __construct(MessengerInterface $messenger) {
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public function checkTaxshelterConfig() {
    $result = [];

    $taxshelter_config = \Drupal::config('shareholder_register_taxshelter.settings');

    $config_url = Url::fromRoute('shareholder_register_taxshelter.shareholder_register_taxshelter_config_form');
    $link = Link::fromTextAndUrl($this->t('Configure it here'), $config_url);

    if (!trim($taxshelter_config->get('signee_name'))) {
      $result[] = [
        'error', $this->t('Signee name is not set! (@link)', [
          '@link' => $link->toString(),
        ]),
      ];
    }
    if (!trim($taxshelter_config->get('signee_function'))) {
      $result[] = [
        'error', $this->t('Signee function is not set! (@link)', [
          '@link' => $link->toString(),
        ]),
      ];
    }
    if (!trim($taxshelter_config->get('signee_phone'))) {
      $result[] = [
        'error', $this->t('Signee phone is not set! (@link)', [
          '@link' => $link->toString(),
        ]),
      ];
    }

    if (!trim($taxshelter_config->get('company_type'))) {
      $result[] = [
        'error', $this->t('Taxshelter company type function is not set! (@link)', [
          '@link' => $link->toString(),
        ]),
      ];
    }
    if (!trim($taxshelter_config->get('date_of_birth_field')) && !trim($taxshelter_config->get('rrn_field'))) {
      $result[] = [
        'error', $this->t('At least one of RRN field and Date of Birth field have to be set! (@link)', [
          '@link' => $link->toString(),
        ]),
      ];
    }

    $config = \Drupal::config('shareholder_register.settings');
    if ($config->get('company_date_of_incorporation') < '2013-01-01') {
      $result[] = [
        'error', $this->t('Taxshelter is only applicable if date of incorporation is after 2012-12-31! (@link)', [
          '@link' => $link->toString(),
        ]),
      ];
    }

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public static function getTaxshelterKbo() {
    $config = \Drupal::config('shareholder_register.settings');
    return preg_replace('/[^0-9]/', '', $config->get('company_id'));
  }

  /**
   * {@inheritdoc}
   */
  public static function getTaxshelterSignature() {
    $config = \Drupal::config('shareholder_register_taxshelter.settings');
    $fid = $config->get('signature_image');
    return count($fid) ? File::load(reset($fid)) : FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public static function getLatestTaxshelterDate() {
    $redemption_types = array_keys(
      \Drupal::entityQuery('share_transaction_group_type')
        ->condition('baseType', 'redemption')
        ->execute()
    );

    $config = \Drupal::config('shareholder_register.settings');
    $first_redemption_id = \Drupal::entityQuery('share_transaction')
      ->condition('state', 'valid')
      ->condition('quantity', 0, '<')
      ->condition('transaction_group.entity.type', $redemption_types, 'IN')
      ->sort('date')->range(0, 1)
      ->execute();
    $first_redemption_date = count($first_redemption_id) ? ShareTransaction::load(
      reset($first_redemption_id))->getDate() : FALSE;

    return min(
      array_filter([
          date('Y-m-d', strtotime(
              '+4 years',
              strtotime($config->get('company_date_of_incorporation')))),
          $first_redemption_date,
    ]));
  }

  /**
   * {@inheritdoc}
   */
  public static function getTotalTaxshelter() {
    $total = '0';
    // Compute total taxshelter value.
    foreach (TaxshelterClaim::loadMultiple(\Drupal::entityQuery('taxshelter_claim')
        ->condition('type', 'verwerving')
        ->execute()) as $claim) {
      $total = bcadd(
        $total,
        $claim->getAmount(),
        2);
    }
    return $total;
  }

  /**
   * {@inheritdoc}
   */
  public static function getBasePdf() {
    $pdf = new Pdf(array(
      'no-outline',
      'margin-top' => 0,
      'margin-right' => 0,
      'margin-bottom' => 0,
      'margin-left' => 0,
      // Default page options.
      'disable-smart-shrinking',
    ));
    return $pdf;
  }

  /**
   * {@inheritdoc}
   */
  public function getTaxshelterBatch($year, $max_date = FALSE) {
    /*
    $entity_ids = \Drupal::entityQuery('taxshelter_claim')
      ->condition('year', $year)
      ->execute();

    if (count($entity_ids)) {
      error_log("There are already taxshelter claims for the year '{$year}'!");
      return [];
    }
    */

    $batch = array(
      'title' => t('Generating documents'),
      'operations' => [],
      'finished' => [get_class($this), 'downloadPdf'],
    );

    $batch['operations'][] = [
      [get_class($this), 'addTaxshelterForShareTransactionBatch'],
      [
        $year,
        $max_date,
      ],
    ];
    $batch['operations'][] = [
      [get_class($this), 'addTaxshelterForClaimsBatch'],
      [
        $year,
      ],
    ];
    $batch['operations'][] = [
      [get_class($this), 'generateCertificatesBatch'],
      [],
    ];

    return $batch;
  }

  /**
   * {@inheritdoc}
   */
  public static function downloadPdf($success, $results, $operations) {
    // FIXME: find another way to invalidate shareholder cache !
    drupal_flush_all_caches();

    if ($success) {
      \Drupal::service('messenger')->addMessage(t('Succesfully generated @number taxshelter documents!', [
        '@number' => $results['count'],
      ]));
    }
    else {
      \Drupal::service('messenger')->addError(t('Error generating the taxshelter documents!'));
    }
  }


  /**
   * {@inheritdoc}
   */
  public static function addTaxshelterForClaimsBatch($year, &$context) {
    if (!isset($context['results']['claims'])) {
      $context['results']['claims'] = [];
    }

    if (!isset($context['sandbox']['claim_ids'])) {
      $start_date = "{$year}-01-01";
      $end_date = "{$year}-12-31";

      $claim_ids = \Drupal::EntityQuery('taxshelter_claim')
        ->condition('year', $year - 1)
        ->condition('type', ['verwerving', 'behoud'], 'IN')
        ->execute();

      $context['sandbox']['claim_ids'] = $claim_ids;
      $context['sandbox']['claim_ids_count'] = count($context['sandbox']['claim_ids']);
    }
    for ($i = 0; $i < 20; $i++) {
      if ($claim_id = array_pop($context['sandbox']['claim_ids'])) {
        $claims = ShareholderRegisterTaxshelterService::addTaxshelterForClaim(
          TaxshelterClaim::load($claim_id),
          $year
        );

        if ($claims) {
          foreach ($claims as $claim) {
            $context['results']['claims'][] = $claim->id();
          }
          $context['results']['count'] += count($claims);
        }

      }
    }
    if ($context['sandbox']['claim_ids_count']) {
      $context['finished'] = ($context['sandbox']['claim_ids_count'] - count($context['sandbox']['claim_ids'])) / $context['sandbox']['claim_ids_count'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function addTaxshelterForClaim($claim, $year) {
    $shareholderRegisterService = \Drupal::service('shareholder_register.default');
    $claims = [];

    if ($claim->getYear() != $year - 1) {
      throw new \Exception("Internal error: Cannot create claim for year '{$year}'" .
        " based on claim '{$claim->id()}' for year '{$claim->getYear()}'!");
    }

    $shareholder = $claim->getShareholder();
    $langcode = "nl";
    if ($shareholder->getPreferredLangcode() === 'fr') {
      $langcode = "fr";
    }

    // No claim if previous claim was alienation.
    if ($claim->getType() === 'vervreemding') {
      return $claims;
    }

    if ($shareholder->get('deceased_date')->value) {
      // No claim year+1 after deceased.
      if (date("Y", strtotime($shareholder->get('deceased_date')->value)) < $year) {
        return $claims;
      }
      $end_date = $shareholder->get('deceased_date')->value;
    }
    else {
      $end_date = "{$year}-12-31";
    }
    $shares = $shareholder->getSharesAtDate($end_date, NULL, TRUE);
    $share_ids = array_map(
      function ($s) {
        return $s->id();
      },
      $shares
    );

    $claim_shares = $claim->get('shares')->referencedEntities();
    $claim_share_ids = array_map(
      function ($s) {
        return $s->id();
      },
      $claim_shares
    );

    $date_of_investment = min(
      array_map(
        function ($s) {
          return $s->getPaymentDate();
        },
        $claim_shares
      )
    );

    // Stop creating claims after 4 years.
    if ($year > (date("Y", strtotime($date_of_investment)) + 4)) {
      return $claims;
    }

    $still_held = array_intersect($claim_share_ids, $share_ids);
    $not_held = array_diff($claim_share_ids, $share_ids);

    if ($still_held) {
      $claim = TaxshelterClaim::create([
        'type' => "behoud",
        'shareholder_id' => $shareholder->id(),
        'year' => $year,
        'amount' => $shareholderRegisterService->sharesTotalValue(
          Share::loadMultiple($still_held)),
        'shares' => $still_held,
      ]);
      $claim->save();
      $claims[] = $claim;
    }
    // FIXME: split by date.
    if ($not_held) {
      $issue_date = Share::load(reset($not_held))->getIssueDate();
      $redeem_date = min(
        array_map(
          function ($s) {
            return $s->getValidTransactions()[1]->getDate();
          },
          Share::loadMultiple($not_held)
        )
      );
      $months_left = 48 - (new \Datetime($redeem_date))->diff(
        new \Datetime($issue_date))->m;

      $claim = TaxshelterClaim::create([
        'type' => "vervreemding",
        'shareholder_id' => $shareholder->id(),
        'year' => $year,
        'amount' => $shareholderRegisterService->sharesTotalValue(
          Share::loadMultiple($not_held)),
        'shares' => $not_held,
        'extra' => [
          'months_left' => $months_left,
        ]
      ]);
      $claim->save();
      $claims[] = $claim;
    }

    return $claims;
  }

  /**
   * {@inheritdoc}
   */
  public static function addTaxshelterForShareTransactionBatch($year, $max_date, &$context) {
    if (!isset($context['results']['claims'])) {
      $context['results']['claims'] = [];
    }

    if (!$max_date) {
      $max_date = ShareholderRegisterTaxshelterService::getLatestTaxshelterDate();
    }

    if (!isset($context['sandbox']['transaction_ids'])) {
      $start_date = "{$year}-01-01";
      $end_date = "{$year}-12-31";

      $transaction_ids = \Drupal::entityQuery('share_transaction')
        ->condition('state', 'valid')
        ->condition('payment_date', $start_date, '>=')
        ->condition('payment_date', $end_date, '<=')
        // Earliest date for taxshelter.
        ->condition('payment_date', '2015-07-01', '>=')
        ->condition('payment_date', $max_date, '<=')
        // Because we use array_pop to process the list.
        ->sort('payment_date', 'DESC')
        ->sort('date', 'DESC')
        ->sort('id', 'DESC')
        ->execute();

      $context['sandbox']['transaction_ids'] = $transaction_ids;
      $context['sandbox']['transaction_ids_count'] = count($context['sandbox']['transaction_ids']);
    }
    for ($i = 0; $i < 20; $i++) {
      if ($transaction_id = array_pop($context['sandbox']['transaction_ids'])) {
        $claim = ShareholderRegisterTaxshelterService::addTaxshelterForShareTransaction(
          ShareTransaction::load($transaction_id),
          $year
        );
        if ($claim) {
          $context['results']['claims'][] = $claim->id();
          $context['results']['count'] += 1;
        }
      }
    }
    if ($context['sandbox']['transaction_ids_count']) {
      $context['finished'] = ($context['sandbox']['transaction_ids_count'] - count($context['sandbox']['transaction_ids'])) / $context['sandbox']['transaction_ids_count'];
    }
  }

  /**
   * {@inheritdoc}
   */
  private static function capAtLimit(&$new_shares, &$new_value, $total, $limit) {
    $shareholderRegisterService = \Drupal::service('shareholder_register.default');

    if ($total >= $limit) {
      return FALSE;
    }
    elseif ($total + $new_value > $limit) {
      $new_value = bcsub($limit, $total, 2);

      $tmp_shares = $new_shares;
      $tmp_total = '0';
      while ($total + $tmp_total < $limit && $s = array_pop($new_shares)) {
        $tmp_shares[] = $s;
        $tmp_total = bcadd(
          $tmp_total,
          $shareholderRegisterService->sharesTotalValue([$s]),
          2);
      }
      $new_shares = $tmp_shares;
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public static function addTaxshelterForShareTransaction($transaction, $year, $force = FALSE, $share_ids = FALSE) {
    $shareholderRegisterService = \Drupal::service('shareholder_register.default');

    $claim = FALSE;
    $shareholder = $transaction->getShareholder();

    // Find newly acquired shares.
    $new_shares = $transaction->getShares();
    $new_value = $shareholderRegisterService->sharesTotalValue($new_shares);

    if (!$force) {

      // Test if shareholder is a natural_person.
      if ($shareholder->bundle() != 'natural_person') {
        error_log("Taxshelter {$year} transaction {$transaction->id()}: skipping, reason: not a natural person");
        return $claim;
      }

      // Test if shareholder has opted out of taxshelter.
      if ($shareholder->hasField('field_no_taxshelter') && $shareholder->get('field_no_taxshelter')->value) {
        error_log("Taxshelter {$year} transaction {$transaction->id()}: skipping, reason: opt-out");
        return $claim;
      }

      // Test if shareholder is belgian.
      if ($shareholder->get('address')->country_code != 'BE' &&
        (!$shareholder->hasField('field_tax_return_non_resident') ||
          !$shareholder->get('field_tax_return_non_resident')->value)) {
        error_log("Taxshelter {$year} transaction {$transaction->id()}: skipping, reason: not BE and not non-resident");
        return $claim;
      }

      // Filter new shares by held at end date.
      $held_shares = $shareholder->getSharesAtDate("{$year}-12-31", NULL, TRUE);
      $held_share_ids = array_map(
        function ($s) {
          return $s->id();
        },
        $held_shares
      );
      $new_shares = array_filter(
        $new_shares,
        function ($s) use ($held_share_ids) {
          return in_array($s->id(), $held_share_ids);
        }
      );
      $new_value = $shareholderRegisterService->sharesTotalValue($new_shares);

      $total = '0';
      $shareholder_total = '0';
      // Compute total taxshelter value.
      foreach (TaxshelterClaim::loadMultiple(\Drupal::entityQuery('taxshelter_claim')
          ->condition('type', 'verwerving')
          ->execute()) as $c) {
        $claim_total = $c->getAmount();
        if ($c->getShareholder()->id() == $shareholder->id() &&
          $c->getYear() == $year) {
          $shareholder_total = bcadd($shareholder_total, $claim_total, 2);
        }
        $total = bcadd($total, $claim_total, 2);
      }

      if (!self::capAtLimit($new_shares, $new_value, $total, self::TAXSHELTER_LIMIT)) {
        error_log("Taxshelter {$year} transaction {$transaction->id()}: skipping, reason: taxshelter limit {self::TAXSHELTER_LIMIT}");
        return $claim;
      }

      if (!self::capAtLimit($new_shares, $new_value, $shareholder_total, self::TAXSHELTER_SHAREHOLDER_LIMIT)) {
        error_log("Taxshelter {$year} transaction {$transaction->id()}: skipping, reason: taxshelter shareholder limit {self::TAXSHELTER_SHAREHOLDER_LIMIT}");
        return $claim;
      }

      $capital_limit = bcmul(
        $shareholderRegisterService->getTotalShareValueAtDate($transaction->getPaymentDate(), NULL, TRUE),
        0.3,
        2
      );

      if (!self::capAtLimit($new_shares, $new_value, $shareholder_total, $capital_limit)) {
        error_log("Taxshelter {$year} transaction {$transaction->id()}: skipping, reason: taxshelter capital limit {$capital_limit}");
        return $claim;
      }
    }

    if ($share_ids) {
      $new_shares = array_filter(
        $new_shares,
        function ($s) use ($share_ids) {
          return in_array($s->id(), $share_ids);
        }
      );
      $new_value = $shareholderRegisterService->sharesTotalValue($new_shares);
    }

    // FIXME: test only 1 transaction in period,
    // which is issue to current holder.
    if ($new_shares) {
      $claim = TaxshelterClaim::create([
        'type' => "verwerving",
        'shareholder_id' => $shareholder->id(),
        'year' => $year,
        'amount' => $new_value,
        'shares' => $new_shares,
      ]);
      $claim->save();
    }

    return $claim;
  }

  /**
   * {@inheritdoc}
   */
  public static function generateCertificatesBatch(&$context) {
    if (!isset($context['sandbox']['claim_ids'])) {
      $context['sandbox']['claim_ids'] = $context['results']['claims'];
      $context['sandbox']['claim_ids_count'] = count($context['sandbox']['claim_ids']);
    }
    for ($i = 0; $i < 20; $i++) {
      if ($claim_id = array_pop($context['sandbox']['claim_ids'])) {
        ShareholderRegisterTaxshelterService::generateCertificate($claim_id);
      }
    }
    if ($context['sandbox']['claim_ids_count']) {
      $context['finished'] = ($context['sandbox']['claim_ids_count'] - count($context['sandbox']['claim_ids'])) / $context['sandbox']['claim_ids_count'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function generateCertificate($claim_id) {
    $claim = TaxshelterClaim::load($claim_id);

    $langcode = "nl";
    if ($claim->getShareholder()->getPreferredLangcode() === 'fr') {
      $langcode = $claim->getShareholder()->getPreferredLangcode();
    }

    $render_array = [
      '#theme' => "shareholder_register_taxshelter_attest_2020_{$langcode}",
      '#document_type' => "verwerving",
      '#shareholder' => $claim->getShareholder(),
      '#share_value' => $claim->getAmount(),
      // '#share_date' => date('d/m/Y', strtotime($payment_date)),
      '#year' => $claim->getYear(),
      // Forms are dutch or french.
      '#print_date' => date('d/m/Y'),
      '#shares' => $claim->getShares(),
      '#claim' => $claim,
    ];

    $mailTemplateService = \Drupal::service('mail_template.default');

    $html = $mailTemplateService->renderRenderArray(
      $mailTemplateService->augmentRenderArray($render_array));

    $pdf = ShareholderRegisterTaxshelterService::getBasePdf();
    $pdf->addPage($html);

    $uri = file_unmanaged_save_data($pdf->toString(), "private://taxshelter-{$claim->getYear()}-{$claim->getShareholder()->getNumber()}.pdf");
    $file = File::create([
      'uri' => $uri,
      'uid' => \Drupal::currentUser()->id(),
      'status' => FILE_STATUS_PERMANENT,
    ]);
    $file->save();

    $claim->set('statement', $file);
    $claim->save();
  }

  /**
   * {@inheritdoc}
   */
  public static function mailTaxshelterForShareholder($shareholder, $year) {
    $email = $shareholder->get('mail')->value;
    $params = [
      'context' => [
        'shareholder' => $shareholder,
        'year' => $year,
      ]
    ];
    $langcode = $shareholder->getPreferredLangcode();
    $mailManager = \Drupal::service('plugin.manager.mail');
    $result = $mailManager->mail('shareholder_register_taxshelter', 'mail_taxshelter', $email, $langcode, $params, NULL, TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public static function generateBelcotaxXml($year, $final) {
    $shareholderRegisterService = \Drupal::service('shareholder_register.default');
    $config = \Drupal::config('shareholder_register.settings');
    $taxshelter_config = \Drupal::config('shareholder_register_taxshelter.settings');

    $kbo = ShareholderRegisterTaxshelterService::getTaxshelterKbo();

    // Retrieve Claims.
    $claim_ids = \Drupal::entityQuery('taxshelter_claim')
      ->condition('year', $year)
      ->execute();

    $service = new XmlService();
    $volgnummer = 1;
    $controle_totaal = '0';

    $fiches = array_map(
      function (
        $entry
      ) use (
        $config,
        $taxshelter_config,
        $kbo,
        $year,
        $shareholderRegisterService,
        &$volgnummer,
        &$controle_totaal
      ) {
        $shareholder = $entry->getShareholder();
        $shares = $entry->getShares();
        $fiche_controle_totaal = 0;

        $reduction = 0;
        if ($taxshelter_config->get('company_type') == 'micro') {
          $reduction = 2;
        }
        elseif ($taxshelter_config->get('company_type') == 'small') {
          $reduction = 1;
        }

        $attest_type = FALSE;
        $attest_data = [];
        if ($entry->getType() == 'verwerving') {
          $attest_type = 1;
          $attest_data = [
            'f85_2032_objectcertificate1' => 1,
            'f85_2060_totalamount' => (int) bcmul($shareholderRegisterService->sharesTotalValue($shares), 100, 2),
            'f85_2061_cutamount' => (int) bcmul($entry->getAmount(), 100, 2),
          ];
          $fiche_controle_totaal += (int) bcmul($shareholderRegisterService->sharesTotalValue($shares), 100, 2);
          $fiche_controle_totaal += (int) bcmul($entry->getAmount(), 100, 2);
        }
        elseif ($entry->getType() == 'behoud') {
          $attest_type = 2;
          $attest_data = [
            // behoud.
            'f85_2035_objectcertificate2' => 1,
          ];
        }
        if ($entry->getType() == 'vervreemding') {
          $attest_type = 3;
          $attest_data = [
            'f85_2090_numbermonth' =>  $entry->getData('months_left'),
          ];
        }

        if ($taxshelter_config->get('rrn_field') &&
          $shareholder->hasField($taxshelter_config->get('rrn_field')) &&
          $shareholder->get($taxshelter_config->get('rrn_field'))->value &&
          strlen($rrn = preg_replace(
              '/[^0-9]/', '', $shareholder->get(
                $taxshelter_config->get('rrn_field'))->value)) == 11) {
          $attest_data['f2011_nationaalnr'] = $rrn;
        }
        if ($taxshelter_config->get('date_of_birth_field') &&
          $shareholder->hasField($taxshelter_config->get('date_of_birth_field')) &&
          $shareholder->get($taxshelter_config->get('date_of_birth_field'))->value) {
          $attest_data['f2012_geboortedatum'] = $shareholder->get(
            $taxshelter_config->get('date_of_birth_field'))->value;
        }

        // Beperkte lijst van RSZ landcodes voorzien, anders manueel aan te passen.
        $land_codes = [
          'FR' => '00111',
          'NL' => '00129',
          'LU' => '00113',
          'DE' => '00103',
          'AT' => '00105',
          'PL' => '00122',
          'SN' => '00320',
          'US' => '00402',
        ];

        if (array_key_exists($shareholder->get('address')->country_code, $land_codes)) {
          $attest_data['f2018_landwoonplaats'] = $land_codes[$shareholder->get('address')->country_code];
          $attest_data['f2112_buitenlandspostnummer'] = $shareholder->getPostalCode();
        }
        elseif ($shareholder->get('address')->country_code == 'BE') {
          $attest_data['f2018_landwoonplaats'] = '00000';
          $attest_data['f2016_postcodebelgisch'] = $shareholder->getPostalCode();
        }
        else {
          $attest_data['f2018_landwoonplaats'] = '';
        }

        $claim_data = [
          'f2002_inkomstenjaar' => $year,
          'f2005_registratienummer' => $kbo,
          'f2008_typefiche' => '28185',
          'f2009_volgnummer' => $volgnummer,
          'f2013_naam' => $shareholder->getFamilyName(),
          'f2015_adres' => $shareholder->getFullAddressLine(),
          'f2017_gemeente' => $shareholder->getLocality(),
          // Normaal (geen wijziging, annulatie).
          'f2028_typetraitement' => 0,
          // Fiche 281.
          'f2029_enkelopgave325' => 0,
          // Rechtstreeks.
          'f2114_voornamen' => $shareholder->getGivenName(),
          'f85_2030_typetaxshelter' => 1,
          'f85_2031_typecertificate' => $attest_type,
          'f85_2034_pctreduction' => $reduction,
          'f85_2056_dateinvesment' => date(
            'd-m-Y', strtotime(reset($shares)->getPaymentDate())),
          'f85_2059_totaalcontrole' => $fiche_controle_totaal,
          // 'f85_2099_addresscompany' => '',
          'f85_2103_certifercompany' => $taxshelter_config->get('signee_name'),
          'f85_2106_functioncertifer' => $taxshelter_config->get('signee_function'),
        ] + $attest_data;

        ksort($claim_data);

        $entry->set('belcotax_serial', $volgnummer);
        $entry->save();
        $volgnummer++;
        $controle_totaal += $fiche_controle_totaal;
        return [
          'Fiche28185' => $claim_data,
        ];
      },
      TaxshelterClaim::loadMultiple($claim_ids)
    );

    $output = $service->write(
      'Verzendingen',
      [
        'Verzending' => [
          'v0002_inkomstenjaar' => $year,
          'v0010_bestandtype' => $final ? 'BELCOTAX' : 'BELCOTST',
          'v0011_aanmaakdatum' => date('d-m-Y'),
          'v0014_naam' => $taxshelter_config->get('signee_name'),
          'v0015_adres' => $config->get('company_address'),
          'v0016_postcode' => $config->get('company_zip'),
          'v0017_gemeente' => $config->get('company_locality'),
          'v0018_telefoonnummer' => $taxshelter_config->get('signee_phone'),
          'v0021_contactpersoon' => $taxshelter_config->get('signee_name'),
          'v0022_taalcode' => 1,
          'v0023_emailadres' => \Drupal::config('system.site')->get('mail'),
          'v0024_nationaalnr' => $kbo,
          'v0025_typeenvoi' => 0,
          'v0028_landwoonplaats' => '00000',
          'Aangiften' => [
            'Aangifte' => [
              'a1002_inkomstenjaar' => $year,
              'a1005_registratienummer' => $kbo,
              'a1011_naamnl1' => $config->get('company_name'),
              'a1013_adresnl' => $config->get('company_address'),
              'a1014_postcodebelgisch' => $config->get('company_zip'),
              'a1015_gemeente' => $config->get('company_locality'),
              'a1016_landwoonplaats' => '00000',
              'a1019_contactpersoon' => $taxshelter_config->get('signee_name'),
              'a1020_taalcode' => 1,
              'Opgaven' => [
                'Opgave32585' => $fiches,
              ],
              'r8002_inkomstenjaar' => $year,
              'r8005_registratienummer' => $kbo,
              'r8010_aantalrecords' => count($claim_ids) + 2,
              'r8011_controletotaal' => (count($claim_ids) * (count($claim_ids) + 1)) / 2,
              'r8012_controletotaal' => $controle_totaal,
            ]
          ],
          'r9002_inkomstenjaar' => $year,
          'r9010_aantallogbestanden' => 1 + 2,
          'r9011_totaalaantalrecords' => count($claim_ids) + 4,
          'r9012_controletotaal' => (count($claim_ids) * (count($claim_ids) + 1)) / 2,
          'r9013_controletotaal' => $controle_totaal,
        ]
      ]
    );

    if ($final) {
      $uri = file_unmanaged_save_data(
        $output,
        "private://belcotax-{$year}.xml"
      );
      $file = File::create([
        'uri' => $uri,
        'uid' => \Drupal::currentUser()->id(),
        'status' => FILE_STATUS_PERMANENT,
      ]);
      $file->save();

      $belcotax = TaxshelterBelcotax::create([
        'name' => $year,
        'claims' => $claim_ids,
      ]);
      $belcotax->set('belcotax_xml', $file);
      $belcotax->save();
    }

    return $output;
  }
}
