<?php

namespace Drupal\Tests\shareholder_register_taxshelter\Kernel;

use Drupal\shareholder_register\Entity\Shareholder;

use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\shareholder_register\Traits\LoadRegisterTrait;

/**
 * Tests taxshelter claim creation.
 *
 * @group shareholder_register
 */
class TaxshelterTest extends KernelTestBase {

  use LoadRegisterTrait;

  /**
   * Modules to enable.
   *
   * @var array
   */
  public static $modules = [
    'system',
    'user',
    'file',
    'text',
    'datetime',
    'simple_workflows',
    'shareholder_register',
    'shareholder_register_taxshelter',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp() {
    parent::setUp();
    $this->traitSetUp();
    $this->installEntitySchema('taxshelter_claim');
  }

  /**
   * Test claim creation sequence.
   */
  public function testClaimSequence() {
    $register = $this->loadBasicRegister(
      [
        'shareholders' => [
          [
            'number' => 1,
          ],
          // Extra shareholder to raise capital.
          [
            'number' => 2,
          ],
        ],
        'transactions' => [
          [
            'quantity' => 5,
            'shareholder_number' => 1,
            'payment_date' => '2017-02-02',
            'validation_date' => '2017-02-02',
          ],
          [
            'quantity' => 50,
            'shareholder_number' => 2,
            'payment_date' => '2016-02-02',
            'validation_date' => '2016-02-02',
          ],
        ],
      ]);

    $t = $register['transactions'][0];

    $c = \Drupal::service('shareholder_register_taxshelter.default')->addTaxshelterForShareTransaction($t, '2017');
    $this->assertEqual($c->getYear(), '2017');
    $this->assertEqual($c->getAmount(), 500);
    $this->assertEqual($c->getType(), 'verwerving');

    $claims = \Drupal::service('shareholder_register_taxshelter.default')->addTaxshelterForClaim($c, '2018');
    $this->assertEqual(count($claims), 1);
    $c2 = reset($claims);
    $this->assertEqual($c2->getYear(), '2018');
    $this->assertEqual($c2->getAmount(), 500);
    $this->assertEqual($c2->getType(), 'behoud');

    $claims = \Drupal::service('shareholder_register_taxshelter.default')->addTaxshelterForClaim($c2, '2019');
    $this->assertEqual(count($claims), 1);
    $c3 = reset($claims);
    $this->assertEqual($c3->getYear(), '2019');
    $this->assertEqual($c3->getAmount(), 500);
    $this->assertEqual($c3->getType(), 'behoud');

    $claims = \Drupal::service('shareholder_register_taxshelter.default')->addTaxshelterForClaim($c3, '2020');
    $this->assertEqual(count($claims), 1);
    $c4 = reset($claims);
    $this->assertEqual($c4->getYear(), '2020');
    $this->assertEqual($c4->getAmount(), 500);
    $this->assertEqual($c4->getType(), 'behoud');

    $claims = \Drupal::service('shareholder_register_taxshelter.default')->addTaxshelterForClaim($c4, '2021');
    $this->assertEqual(count($claims), 1);
    $c5 = reset($claims);
    $this->assertEqual($c5->getYear(), '2021');
    $this->assertEqual($c5->getAmount(), 500);
    $this->assertEqual($c5->getType(), 'behoud');

    $claims = \Drupal::service('shareholder_register_taxshelter.default')->addTaxshelterForClaim($c5, '2022');
    $this->assertEqual(count($claims), 0, "No claim for year + 5");
  }

  /**
   * Test claim creation out-of-sequence sequence.
   */
  public function testClaimOutOfSequenceGeneration() {
    $register = $this->loadBasicRegister(
      [
        'shareholders' => [
          [
            'number' => 1,
          ],
          // Extra shareholder to raise capital.
          [
            'number' => 2,
          ],
        ],
        'transactions' => [
          [
            'quantity' => 5,
            'shareholder_number' => 1,
            'payment_date' => '2017-02-02',
            'validation_date' => '2017-02-02',
          ],
          [
            'quantity' => 50,
            'shareholder_number' => 2,
            'payment_date' => '2016-02-02',
            'validation_date' => '2016-02-02',
          ],
        ],
      ]);

    $t = $register['transactions'][0];

    $c = \Drupal::service('shareholder_register_taxshelter.default')->addTaxshelterForShareTransaction($t, '2017');
    $this->assertEqual($c->getYear(), '2017');
    $this->assertEqual($c->getAmount(), 500);
    $this->assertEqual($c->getType(), 'verwerving');

    $this->expectException(\Exception::class);
    $claims = \Drupal::service('shareholder_register_taxshelter.default')->addTaxshelterForClaim($c, '2019');
  }

  /**
   * Test alienation claims creation.
   */
  public function testTaxshelterAlienation() {
    $register = $this->loadBasicRegister(
      [
        'shareholders' => [
          [
            'number' => 1,
          ],
          // Extra shareholder to raise capital.
          [
            'number' => 2,
          ],
        ],
        'transactions' => [
          [
            'quantity' => 5,
            'shareholder_number' => 1,
            'payment_date' => '2017-02-02',
            'validation_date' => '2017-02-02',
          ],
          [
            'quantity' => -5,
            'shareholder_number' => 1,
            'validation_date' => '2018-02-02',
          ],
          [
            'quantity' => 50,
            'shareholder_number' => 2,
            'payment_date' => '2016-02-02',
            'validation_date' => '2016-02-02',
          ],
        ],
      ]);

    $t = $register['transactions'][0];
    $c = \Drupal::service('shareholder_register_taxshelter.default')->addTaxshelterForShareTransaction($t, '2017');
    $this->assertEqual($c->getYear(), '2017');
    $this->assertEqual($c->getAmount(), 500);
    $this->assertEqual($c->getType(), 'verwerving');

    $claims = \Drupal::service('shareholder_register_taxshelter.default')->addTaxshelterForClaim($c, '2018');
    $this->assertEqual(count($claims), 1);
    $c2 = reset($claims);
    $this->assertEqual($c2->getYear(), '2018');
    $this->assertEqual($c2->getAmount(), 500);
    $this->assertEqual($c2->getType(), 'vervreemding');

    $claims = \Drupal::service('shareholder_register_taxshelter.default')->addTaxshelterForClaim($c2, '2019');
    $this->assertEqual(count($claims), 0, "No claims the year after alienation");
  }

  /**
   * Test no alienation claim creation after deceased.
   */
  public function testTaxshelterNoAlienationAfterDeceased() {
    $register = $this->loadBasicRegister(
      [
        'shareholders' => [
          [
            'number' => 1,
          ],
          // Extra shareholder to raise capital.
          [
            'number' => 2,
          ],
        ],
        'transactions' => [
          [
            'quantity' => 5,
            'shareholder_number' => 1,
            'payment_date' => '2017-02-02',
            'validation_date' => '2017-02-02',
          ],
          [
            'quantity' => -5,
            'shareholder_number' => 1,
            'validation_date' => '2018-02-02',
          ],
          [
            'quantity' => 50,
            'shareholder_number' => 2,
            'payment_date' => '2016-02-02',
            'validation_date' => '2016-02-02',
          ],
        ],
      ]);

    // Set shareholder deceased.
    $s = Shareholder::load(reset($register['shareholder']));
    $s->set('deceased_date', '2018-01-20');
    $s->save();

    $t = $register['transactions'][0];
    $c = \Drupal::service('shareholder_register_taxshelter.default')->addTaxshelterForShareTransaction($t, '2017');
    $this->assertEqual($c->getYear(), '2017');
    $this->assertEqual($c->getAmount(), 500);
    $this->assertEqual($c->getType(), 'verwerving');

    $claims = \Drupal::service('shareholder_register_taxshelter.default')->addTaxshelterForClaim($c, '2018');
    $this->assertEqual(count($claims), 1);
    $c2 = reset($claims);
    $this->assertEqual($c2->getYear(), '2018');
    $this->assertEqual($c2->getAmount(), 500);
    $this->assertEqual($c2->getType(), 'behoud', "Claim type 'behoud' for year deceased");

    $claims = \Drupal::service('shareholder_register_taxshelter.default')->addTaxshelterForClaim($c2, '2019');
    $this->assertEqual(count($claims), 0, "No claim created for year deceased + 1");
  }

  /**
   * Test that no claim is created if shares have been alienated.
   */
  public function testNoClaimForTransactionIfSharesAlienated() {
    $register = $this->loadBasicRegister(
      [
        'shareholders' => [
          [
            'number' => 1,
          ],
        ],
        'transactions' => [
          [
            'quantity' => 5,
            'shareholder_number' => 1,
            'payment_date' => '2019-02-02',
            'validation_date' => '2019-02-02',
          ],
          [
            'quantity' => -5,
            'shareholder_number' => 1,
            'validation_date' => '2019-12-31',
          ],
        ],
      ]);

    $t = $register['transactions'][0];
    $c = \Drupal::service('shareholder_register_taxshelter.default')->addTaxshelterForShareTransaction($t, '2019');
    $this->assertEqual($c, FALSE, "No claim if shares have been alienated before end of year");
  }

  /**
   * Test that no claim is created if shares have been alienated.
   */
  /*
  public function testNoClaimForTransactionIfNotNaturalPerson() {
    $register = $this->loadBasicRegister(
      [
        'shareholders' => [
          [
            'number' => 1,
            'type' => 'legal_person',
          ],
        ],
        'transactions' => [
          [
            'quantity' => 5,
            'shareholder_number' => 1,
            'payment_date' => '2019-02-02',
            'validation_date' => '2019-02-02',
          ],
        ],
      ]);

    $t = $register['transactions'][0];
    $this->expectError();
    //$this->expectErrorMessage("Taxshelter 2019 transaction 1: skipping, reason: not a natural person");
    $c = \Drupal::service('shareholder_register_taxshelter.default')->addTaxshelterForShareTransaction($t, '2019');
    $this->assertEqual($c, FALSE, "No claim if shareholder is not a natural person.");
  }
  */

  /**
   * Test that claims per shareholder are capped at 30% of capital.
   */
  public function testShareholderCapitalLimit() {
    $register = $this->loadBasicRegister(
      [
        'shareholders' => [
          [
            'number' => 1,
          ],
        ],
        'transactions' => [
          [
            'quantity' => 5,
            'shareholder_number' => 1,
            'payment_date' => '2019-02-02',
            'validation_date' => '2019-02-02',
          ],
        ],
      ]);

    $t = $register['transactions'][0];
    $c = \Drupal::service('shareholder_register_taxshelter.default')->addTaxshelterForShareTransaction($t, '2019');
    $this->assertEqual($c->getYear(), '2019');
    $this->assertEqual($c->getAmount(), 150);
  }

  /**
   * Tests that claims per shareholder are capped at 100000.
   */
  public function testShareholderLimit() {
    $register = $this->loadBasicRegister(
      [
        'shareholders' => [
          [
            'number' => 1,
          ],
          [
            'number' => 2,
          ],
        ],
        'transactions' => [
          [
            'quantity' => 50,
            'shareholder_number' => 1,
            'payment_date' => '2019-02-02',
            'validation_date' => '2019-02-02',
          ],
          [
            'quantity' => 11,
            'shareholder_number' => 1,
            'payment_date' => '2019-02-03',
            'validation_date' => '2019-02-03',
          ],
        ],
      ],
      10000
    );

    $t = $register['transactions'][1];
    $c = \Drupal::service('shareholder_register_taxshelter.default')->addTaxshelterForShareTransaction($t, '2019');
    $this->assertEqual($c->getYear(), '2019');
    $this->assertEqual($c->getAmount(), 100000);
  }

  /**
   * Test that taxshelter claims are capped at 250000.
   */
  public function testTaxshelterLimit() {
    $register = $this->loadBasicRegister(
      [
        'shareholders' => [
          [
            'number' => 1,
          ],
          [
            'number' => 2,
          ],
          [
            'number' => 3,
          ],
          [
            'number' => 4,
          ],
        ],
        'transactions' => [
          [
            'quantity' => 50,
            'shareholder_number' => 1,
            'payment_date' => '2019-02-02',
            'validation_date' => '2019-02-02',
          ],
          [
            'quantity' => 10,
            'shareholder_number' => 2,
            'payment_date' => '2019-02-03',
            'validation_date' => '2019-02-03',
          ],
          [
            'quantity' => 10,
            'shareholder_number' => 3,
            'payment_date' => '2019-02-03',
            'validation_date' => '2019-02-03',
          ],
          [
            'quantity' => 10,
            'shareholder_number' => 4,
            'payment_date' => '2019-02-03',
            'validation_date' => '2019-02-03',
          ],
        ],
      ],
      10000
    );

    $c = \Drupal::service('shareholder_register_taxshelter.default')->addTaxshelterForShareTransaction(
      $register['transactions'][1], '2019');
    $c = \Drupal::service('shareholder_register_taxshelter.default')->addTaxshelterForShareTransaction(
      $register['transactions'][2], '2019');
    $c = \Drupal::service('shareholder_register_taxshelter.default')->addTaxshelterForShareTransaction(
      $register['transactions'][3], '2019');

    $this->assertEqual($c->getYear(), '2019');
    $this->assertEqual($c->getAmount(), 50000);
  }

}
