<?php

namespace Drupal\shareholder_register_vote_weight;

use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Reader\Xlsx;

use Drupal\Core\Url;
use Drupal\Core\Link;
use Drupal\Core\StringTranslation\StringTranslationTrait;

use Drupal\shareholder_register\Entity\Shareholder;

/**
 * Service to export Voting Weights for the Shareholder General Meeting.
 */
class GeneralMeetingService implements GeneralMeetingServiceInterface {
  use StringTranslationTrait;

  /**
   * Constructs a new GeneralMeetingService object.
   */
  public function __construct() {
  }

  /**
   * {@inheritdoc}
   */
  public function getGeneralMeetingBatch($config) {
    $batch = [
      'title' => $this->t('Computing general meeting voting weight'),
      'operations' => [],
    ];
    $batch['operations'][] = [
      '\Drupal\shareholder_register_vote_weight\GeneralMeetingService::initGeneralMeetingBatch',
      [$config],
    ];
    $batch['operations'][] = [
      '\Drupal\shareholder_register_vote_weight\GeneralMeetingService::computeGeneralMeetingBatch',
      [$config],
    ];
    $batch['operations'][] = [
      '\Drupal\shareholder_register_vote_weight\GeneralMeetingService::writeGeneralMeetingBatch',
      [$config],
    ];
    $batch['operations'][] = [
      '\Drupal\shareholder_register_vote_weight\GeneralMeetingService::writeGeneralMeetingDetailsBatch',
      [$config],
    ];
    $batch['finished'] = '\Drupal\shareholder_register_vote_weight\GeneralMeetingService::finishGeneralMeetingBatch';

    return $batch;
  }

  /**
   * {@inheritdoc}
   */
  public static function initGeneralMeetingBatch($general_meeting_config, &$context) {
    $context['results']['details'] = [];
    $context['results']['totals'] = [];
    $context['results']['config'] = $general_meeting_config;
  }

  /**
   * {@inheritdoc}
   */
  public static function getShareHash($share) {
    return [
      'hash' => $share->getShareType()->id(),
      'label' => $share->getShareType()->label(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function computeGeneralMeetingBatch($general_meeting_config, &$context) {
    $formatter = \Drupal::service('shareholder_register.formatter');

    if (!isset($context['sandbox']['shareholder_ids'])) {
      $shareholder_ids = \Drupal::entityQuery('shareholder')
        ->condition('state', 'valid')
        ->execute();

      $context['sandbox']['shareholder_ids'] = $shareholder_ids;
      $context['sandbox']['shareholder_ids_count'] = count($context['sandbox']['shareholder_ids']);
      $context['results']['totals'] = [];
      $context['results']['details'] = [];
    }

    for ($i = 0; $i < 100; $i++) {
      if ($shareholder_id = array_pop($context['sandbox']['shareholder_ids'])) {
        $shareholder = Shareholder::load($shareholder_id);

        $shares = $shareholder->getSharesAtDate($general_meeting_config['date']);

        if (!count($shares)) {
          continue;
        }
        $weight = 0;
        $shares_by_hash = [];
        foreach ($shares as $share) {
          $hash_detail = static::getShareHash($share);
          $weight += $share->getShareType()->get('vote_weight')->value;

          if (!isset($context['results']['details'][$shareholder->id()])) {
            $context['results']['details'][$shareholder->id()] = [];
          }
          if (!isset($context['results']['details'][$shareholder->id()][$hash_detail['hash']])) {
            $context['results']['details'][$shareholder->id()][$hash_detail['hash']] = [
              'shareholder_id' => $shareholder->id(),
              'share_numbers' => [],
              'label' => $hash_detail['label'],
              'weight' => 0,
            ];
            $shares_by_hash[$hash_detail['hash']] = [];
          }
          $context['results']['details'][$shareholder->id()][$hash_detail['hash']]['weight'] += $share->getShareType()->get('vote_weight')->value;
          $shares_by_hash[$hash_detail['hash']][] = $share;
        }

        foreach ($shares_by_hash as $hash => $shares_for_hash) {
          $context['results']['details'][$shareholder->id()][$hash]['shares'] = $formatter->sharesToRanges($shares_for_hash);
        }

        $context['results']['totals'][$shareholder->id()] = [
          'shareholder_id' => $shareholder->id(),
          'shares' => $formatter->sharesToRanges($shares),
          'weight' => $weight,
        ];

      }
    }

    if ($context['sandbox']['shareholder_ids_count'] > 0) {
      $context['finished'] = ($context['sandbox']['shareholder_ids_count'] - count($context['sandbox']['shareholder_ids'])) / $context['sandbox']['shareholder_ids_count'];
    }
    else {
      $context['finished'] = 1;
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function writeGeneralMeetingBatch($general_meeting_config, &$context) {
    if (!isset($context['results']['output'])) {
      $output = file_save_data('', 'private://general_meeting.xlsx');
      $output->setTemporary();
      $output->save();

      $context['results']['output'] = $output;

      $spreadsheet = new Spreadsheet();

      // Set document properties.
      $spreadsheet->getProperties()->setCreator('DSR')
        ->setLastModifiedBy(t('DSR'))
        ->setTitle(t('DSR General Meeting Voting Weight'))
        ->setSubject(t('DSR General Meeting Voting Weight'))
        ->setDescription(t('DSR General Meeting Voting Weight'))
        ->setCategory(t('DSR General Meeting'));
    }
    else {
      $reader = new Xlsx();
      $reader->setReadDataOnly(TRUE);
      $spreadsheet = $reader->load(drupal_realpath($context['results']['output']->getFileUri()));
    }

    if (!isset($context['sandbox']['to_write'])) {
      $sheet = $spreadsheet->setActiveSheetIndex(0);

      $sheet->setCellValue("A1", t('General Meeting Voting Weight for date @date',
      [
        '@date' => $general_meeting_config['date'],
      ]));
      $sheet->setCellValue("A3", t('Shareholder Number'));
      $sheet->setCellValue("B3", t('Shareholder Name'));
      $sheet->setCellValue("C3", t('Shares'));
      $sheet->setCellValue("D3", t('Voting weight'));

      $context['sandbox']['row'] = 5;
      $context['sandbox']['to_write'] = $context['results']['totals'];
    }
    else {
      $sheet = $spreadsheet->getActiveSheet();
    }

    for ($i = 0; $i < 5000; $i++) {
      if ($group = array_pop($context['sandbox']['to_write'])) {
        $row = $context['sandbox']['row'];
        $shareholder = Shareholder::load($group['shareholder_id']);

        $sheet->setCellValue("A{$row}", $shareholder->getNumber());
        $sheet->setCellValue("B{$row}", $shareholder->getName());
        $sheet->setCellValue("C{$row}", $group['shares']);
        $sheet->setCellValue("D{$row}", $group['weight']);

        $context['sandbox']['row']++;
      }
    }

    $writer = IOFactory::createWriter($spreadsheet, 'Xlsx');
    $writer->setPreCalculateFormulas(FALSE);
    $writer->save(drupal_realpath($context['results']['output']->getFileUri()));
    $context['results']['output']->save();
    $context['finished'] = (count($context['results']['totals']) - count($context['sandbox']['to_write'])) / count($context['results']['totals']);
  }

  /**
   * {@inheritdoc}
   */
  public static function writeGeneralMeetingDetailsBatch($general_meeting_config, &$context) {
    if (!isset($context['results']['output'])) {
      $output = file_save_data('', 'private://general_meeting.xlsx');
      $output->setTemporary();
      $output->save();

      $context['results']['output'] = $output;

      $spreadsheet = new Spreadsheet();

      // Set document properties.
      $spreadsheet->getProperties()->setCreator('DSR')
        ->setLastModifiedBy(t('DSR'))
        ->setTitle(t('DSR General Meeting Voting Weight'))
        ->setSubject(t('DSR General Meeting Voting Weight'))
        ->setDescription(t('DSR General Meeting Voting Weight'))
        ->setCategory(t('DSR General Meeting'));
    }
    else {
      $reader = new Xlsx();
      $reader->setReadDataOnly(TRUE);
      $spreadsheet = $reader->load(drupal_realpath($context['results']['output']->getFileUri()));
    }

    if (!isset($context['sandbox']['to_write'])) {
      $spreadsheet->createSheet();
      $sheet = $spreadsheet->setActiveSheetIndex(1);

      $sheet->setCellValue("A1", t('General Meeting Voting Weight for date @date',
      [
        '@date' => $general_meeting_config['date'],
      ]));
      $sheet->setCellValue("A3", t('Shareholder Number'));
      $sheet->setCellValue("B3", t('Shareholder Name'));
      $sheet->setCellValue("C3", t('Type'));
      $sheet->setCellValue("D3", t('Shares'));
      $sheet->setCellValue("E3", t('Voting weight'));

      $context['sandbox']['row'] = 5;
      $context['sandbox']['to_write'] = $context['results']['details'];
    }
    else {
      $sheet = $spreadsheet->getActiveSheet();
    }

    for ($i = 0; $i < 5000; $i++) {
      if ($shareholder_group = array_pop($context['sandbox']['to_write'])) {
        foreach ($shareholder_group as $group) {
          $row = $context['sandbox']['row'];
          $shareholder = Shareholder::load($group['shareholder_id']);

          $sheet->setCellValue("A{$row}", $shareholder->getNumber());
          $sheet->setCellValue("B{$row}", $shareholder->getName());
          $sheet->setCellValue("C{$row}", $group['label']);
          $sheet->setCellValue("D{$row}", $group['shares']);
          $sheet->setCellValue("E{$row}", $group['weight']);

          $context['sandbox']['row']++;
        }
      }
    }

    $writer = IOFactory::createWriter($spreadsheet, 'Xlsx');
    $writer->setPreCalculateFormulas(FALSE);
    $writer->save(drupal_realpath($context['results']['output']->getFileUri()));
    $context['results']['output']->save();
    $context['finished'] = (count($context['results']['totals']) - count($context['sandbox']['to_write'])) / count($context['results']['totals']);
  }

  /**
   * {@inheritdoc}
   */
  public static function finishGeneralMeetingBatch($success, $results, $operations) {
    drupal_set_message(t(
        'Exported General Meeting Voting Weight  download: @url',
        [
          '@url' => Link::fromTextAndUrl('general_meeting.xlsx', Url::fromUri(file_create_url($results['output']->getFileUri())))->toString(),
        ]
    ));
  }

}
