<?php

namespace Drupal\shareholder_register_webform\Element;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\HtmlCommand;

use Drupal\Component\Utility\Html;
use Drupal\webform\Element\WebformCompositeBase;
use Drupal\Core\Form\FormStateInterface;

use Drupal\shareholder_register\Entity\ShareIssue;
use Drupal\shareholder_register\Entity\ShareTransaction;

use Drupal\shareholder_register_limit\Exception\ShareTransactionQuantityExceedsLimitException;

/**
 * Provides a 'shares'.
 *
 * @FormElement("shareholder_shares")
 *
 * @see \Drupal\webform\Element\WebformCompositeBase
 * @see \Drupal\shareholder_register_webform\Element\Shares
 */
class ShareholderShares extends WebformCompositeBase {

  /**
   * {@inheritdoc}
   */
  public function getInfo() {
    $return = parent::getInfo() + [
      '#theme' => 'shareholder_shares',
      '#element_validate' => [[get_class($this), 'validateShareholderShares']],
    ];
    $return['#pre_render'][] = [
      get_class($this),
      'preRenderShareholderShares',
    ];
    return $return;
  }

  /**
   * {@inheritdoc}
   */
  public static function getCompositeElements(array $element) {
    // Generate an unique ID that can be used by #states.
    $html_id = Html::getUniqueId('shareholder-shares');

    $options = [];
    foreach (\Drupal::entityTypeManager()->getListBuilder('share_issue')->load() as $share_issue) {
      $options[$share_issue->id()] = $share_issue->label();
    }

    $elements = [
      'share_issue' => [
        '#type' => 'select',
        '#title' => t('Share Issue'),
        '#options' => $options,
        '#attributes' => ['data-webform-shareholder-shares-id' => $html_id . '--share-issue'],
      ],
      'share_quantity' => [
        '#type' => 'number',
        '#min' => 1,
        '#title' => t('Quantity'),
        '#attributes' => ['data-webform-shareholder-shares-id' => $html_id . '--share-quantity'],
        '#ajax' => [
          'callback' => [get_called_class(), 'updatePrice'],
          'event' => 'change',
        ],
        '#suffix' => "<div id=\"share-total-price\"></div>",
      ],
      'share_price' => [
        '#type' => 'value',
        '#title' => t('Share Price'),
      ],
      'total_amount' => [
        '#type' => 'value',
        '#title' => t('Total amount'),
      ]
    ];
    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public static function preRenderShareholderShares($element) {
    if (isset($element['#default_value']) && isset($element['#default_value']['share_issue'])
      && isset($element['#default_value']['share_quantity'])) {
      $share_total_price = ShareholderShares::computeSharePrice(
        $element['#default_value']['share_issue'],
        $element['#default_value']['share_quantity']
      );
      $element['share_quantity']['#suffix'] = "<div id=\"share-total-price\">{$share_total_price}</div>";
    }
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  private static function getSharePrice($share_issue_id, $qty) {
    $share_issue = ShareIssue::load($share_issue_id);

    $comp = [];
    $unit = 0;
    $total = 0;

    if ($share_issue) {
      $comp = $share_issue->getSharePriceComponents();
      $unit = $share_issue->getSharePrice();
      $total = $unit * $qty;
    }

    return [
      'components' => $comp,
      'unit' => $unit,
      'total' => $total,
      'qty' => $qty,
      'type' => $share_issue_id,
    ];
  }

  /**
   * {@inheritdoc}
   */
  private static function computeSharePrice($share_type_id, $qty) {
    $data = ShareholderShares::getSharePrice(
      $share_type_id,
      $qty
    );

    if (empty($data)) {
      return '';
    }

    $render = [
      'price' => [
        '#theme' => 'webform_share_transaction_price',
        '#price_elements' => $data['components'],
        '#quantity' => $data['qty'],
        '#total' => $data['total'],
      ]
    ];

    return drupal_render($render);
  }

  /**
   * {@inheritdoc}
   */
  public static function updatePrice(array $form, FormStateInterface $form_state) {
    $qty_key = $form_state->getTriggeringElement()['#parents'];
    $share_issue_key = array_slice($qty_key, 0, -1);
    $share_issue_key[] = 'share_issue';

    $share_issue = $form_state->getValue($share_issue_key);
    $qty = $form_state->getValue($qty_key);

    $text = ShareholderShares::computeSharePrice(
      $share_issue,
      $qty
    );
    $ajax_response = new AjaxResponse();
    $ajax_response->addCommand(new HtmlCommand('#share-total-price', $text));
    return $ajax_response;
  }

  /**
   * {@inheritdoc}
   */
  public static function validateShareholderShares(&$element, FormStateInterface $form_state, &$complete_form) {
    $parents = $element['#parents'];
    $total_amount_key = array_merge($parents, ['total_amount']);
    $share_price_key = array_merge($parents, ['share_price']);
    $values = $form_state->getValue($parents);

    $share_issue = $values['share_issue'];
    if (!$share_issue) {
      $share_issue_key = $parents;
      $share_issue_key[] = 'share_issue';
      $share_issue = $element['#default_value']['share_issue'];
      $form_state->setValue($share_issue_key, $share_issue);
    }

    if (!$share_issue) {
      $form_state->setError($element, t('Share Issue is required!'));
      return;
    }

    $data = ShareholderShares::getSharePrice(
      $share_issue,
      $values['share_quantity']
    );

    $shareIssue = ShareIssue::load($share_issue);
    $transaction_data = [
      'share_issue_id' => $share_issue,
      'share_type_id' => $shareIssue->getShareType()->id(),
      'quantity' => $values['share_quantity'],
    ];

    if (isset($element['#shareholder_number_key']) &&
      $form_state->getValue([$element['#shareholder_number_key'], 'shareholder_id'])) {
      $transaction_data['shareholder_id'] = $form_state->getValue(
        [$element['#shareholder_number_key'], 'shareholder_id']);
    }

    $t = ShareTransaction::create($transaction_data);
    try {
      $t->preValidateTransaction(date("Y-m-d"));
    }
    catch (ShareTransactionQuantityExceedsLimitException $e) {
      $form_state->setError(
        $element,
        t('You can sign up for at most @max_shares shares! (@error)',
          $e->getMsgPlaceholders() + ['@error' => $e->getMessage()]
        ));
    }
    /*
    catch (\Exception $e) {
      $form_state->setError($element, t('You can sign up for at most @number shares!', [
            '@number' => ,
            // '@number' => $res['remaining_shares'],
      ]));
    }
    */

    $form_state->setValue($total_amount_key, $data['total']);
    $form_state->setValue($share_price_key, $data['unit']);

    if ($values['share_quantity'] <= 0 || $data['total'] <= 0) {
      $form_state->setError(
        $element, t('Invalid number of shares or total amount!')
      );
    }
  }
}
