<?php
use Drupal\shareholder_register\Entity\ShareTransaction;
use Drupal\shareholder_register\Entity\Shareholder;
use Drupal\shareholder_register\Entity\ShareIssue;
use Drupal\shareholder_register\Entity\ShareType;

use Drupal\webform\Entity\Webform;
use Drupal\webform\Utility\WebformElementHelper;

use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Create share issues for existing share types.
 */
function shareholder_register_post_update_issues(&$sandbox) {

  $issues = [];
  foreach (ShareIssue::loadMultiple() as $si) {
    if ($si->get('share_type_id')->target_id) {
      $issues[$si->get('share_type_id')->target_id] = $si;
    }
  }

  foreach (ShareType::loadMultiple() as $st) {
    if (!array_key_exists($st->id(), $issues)) {
      $si = ShareIssue::create([
        'share_type_id' => $st->id(),
        'share_value' => $st->getParValue(),
        'issue_premium' => $st->getIssuePremium(),
      ]);
      $si->save();
      $issues[$st->id()] = $si;
    }
  }

  foreach (ShareTransaction::loadMultiple() as $tr) {
    if ($tr->getQuantity() > 0 && !$tr->getShareIssue()) {
      $tr->set('share_issue_id', $issues[$tr->getShareType()->id()]);
      $tr->save();
    }
  }

  foreach (Webform::loadMultiple() as $w) {
    $shares_key = FALSE;
    foreach (WebformElementHelper::getFlattened($w->getElementsDecoded()) as $k => $e) {
      if ($e['#type'] == 'shareholder_shares') {
        $shares_key = $k;
      }
    }
    if ($shares_key) {
      foreach (\Drupal::entityTypeManager()->getStorage('webform_submission')->loadByProperties([
        'webform_id' => $w->id(),
      ]) as $s) {
        $s_data = $s->getElementData($shares_key);
        if (!$s_data['share_type'] || !array_key_exists($s_data['share_type'], $issues)) {
          error_log("Not updating submission {$s->id()}: no issue found!");
          continue;
        }
        $s_data['share_issue'] = $issues[$s_data['share_type']]->id();
        $s->setElementData($shares_key, $s_data);
        $s->save();
      }
    }
  }

}

/**
 * Create share entities for existing transactions.
 */
function shareholder_register_post_update_shares(&$sandbox) {
  \Drupal::entityDefinitionUpdateManager()->applyUpdates();

  $connection = \Drupal::database();
  $result = $connection->query("select id from share_transaction where state = 'valid' order by convert(name, unsigned integer)");

  foreach (ShareTransaction::loadMultiple($result->fetchAll(PDO::FETCH_COLUMN, 0)) as $t) {
    $t->attachSharesToTransaction($t->getDate());
    $t->save();
  }
}

/**
 * Update current status for shareholders.
 */
function shareholder_register_post_update_current(&$sandbox) {
  foreach (Shareholder::loadMultiple() as $s) {
    $s->save();
  }
}

/**
 * Add unique indexes.
 */
function shareholder_register_post_update_indexes(&$sandbox) {
  $schema = \Drupal::database()->schema();
  if (!$schema->indexExists('shareholder', 'shareholder_number')) {
    $schema->addUniqueKey('shareholder', 'shareholder_number', ['number']);
  }
  if (!$schema->indexExists('share', 'share_name')) {
    $schema->addUniqueKey('share', 'share_name', ['name']);
  }
  if (!$schema->indexExists('share_transaction', 'share_transaction_name')) {
    $schema->addUniqueKey('share_transaction', 'share_transaction_name', ['name']);
  }
  if (!$schema->indexExists('share_transaction_group', 'share_transaction_group_number')) {
    $schema->addUniqueKey('share_transaction_group', 'share_transaction_group_number', ['number']);
  }
}

/**
 * Make share revisionable.
 */
function shareholder_register_post_update_make_share_revisionable(&$sandbox) {
  $definition_update_manager = \Drupal::entityDefinitionUpdateManager();
  /** @var \Drupal\Core\Entity\EntityLastInstalledSchemaRepositoryInterface $last_installed_schema_repository */
  $last_installed_schema_repository = \Drupal::service('entity.last_installed_schema.repository');

  // $definition = $definition_update_manager->getFieldStorageDefinition('langcode', 'share');
  // $definition_update_manager->uninstallFieldStorageDefinition($definition);

  $entity_type = $definition_update_manager->getEntityType('share');
  $field_storage_definitions = $last_installed_schema_repository->getLastInstalledFieldStorageDefinitions('share');

  // Update the entity type definition.
  $entity_keys = $entity_type->getKeys();
  $entity_keys['revision'] = 'vid';
  // unset($entity_keys['langcode']);
  // $entity_keys['revision_translation_affected'] = 'revision_translation_affected';
  $entity_type->set('entity_keys', $entity_keys);
  $entity_type->set('revision_table', 'share_revision');
  $entity_type->set('revision_data_table', 'share_field_revision');
  $revision_metadata_keys = [
    'revision_default' => 'revision_default',
    'revision_user' => 'revision_user',
    'revision_created' => 'revision_created',
    'revision_log_message' => 'revision_log_message',
    'revision_date' => 'revision_date',
  ];
  $entity_type->set('revision_metadata_keys', $revision_metadata_keys);

  // Update the field storage definitions and add the new ones required by a
  // revisionable entity type.
  $field_storage_definitions['user_id']->setRevisionable(TRUE);
  $field_storage_definitions['share_type_id']->setRevisionable(TRUE);
  $field_storage_definitions['state']->setRevisionable(TRUE);

  $field_storage_definitions['vid'] = BaseFieldDefinition::create('integer')
    ->setName('vid')
    ->setTargetEntityTypeId('share')
    ->setTargetBundle(NULL)
    ->setLabel(new TranslatableMarkup('Revision ID'))
    ->setReadOnly(TRUE)
    ->setSetting('unsigned', TRUE);

  $field_storage_definitions['revision_default'] = BaseFieldDefinition::create('boolean')
    ->setName('revision_default')
    ->setTargetEntityTypeId('share')
    ->setTargetBundle(NULL)
    ->setLabel(new TranslatableMarkup('Default revision'))
    ->setDescription(new TranslatableMarkup('A flag indicating whether this was a default revision when it was saved.'))
    ->setStorageRequired(TRUE)
    ->setInternal(TRUE)
    ->setRevisionable(TRUE);

  $field_storage_definitions['revision_created'] = BaseFieldDefinition::create('created')
    ->setName('revision_created')
    ->setTargetEntityTypeId('share')
    ->setTargetBundle(NULL)
    ->setLabel(new TranslatableMarkup('Revision create time'))
    ->setDescription(new TranslatableMarkup('The time that the current revision was created.'))
    ->setRevisionable(TRUE);

  $field_storage_definitions['revision_user'] = BaseFieldDefinition::create('entity_reference')
    ->setName('revision_user')
    ->setTargetEntityTypeId('share')
    ->setTargetBundle(NULL)
    ->setLabel(new TranslatableMarkup('Revision user'))
    ->setDescription(new TranslatableMarkup('The user ID of the author of the current revision.'))
    ->setSetting('target_type', 'user')
    ->setRevisionable(TRUE);

  $field_storage_definitions['revision_log_message'] = BaseFieldDefinition::create('string_long')
    ->setName('revision_log_message')
    ->setTargetEntityTypeId('share')
    ->setTargetBundle(NULL)
    ->setLabel(new TranslatableMarkup('Revision log message'))
    ->setDescription(new TranslatableMarkup('Briefly describe the changes you have made.'))
    ->setRevisionable(TRUE)
    ->setDefaultValue('');

  $field_storage_definitions['revision_date'] = BaseFieldDefinition::create('datetime')
    ->setName('revision_date')
    ->setTargetEntityTypeId('share')
    ->setTargetBundle(NULL)
    ->setLabel(new TranslatableMarkup('Revision date'))
    ->setDescription(new TranslatableMarkup('The date at which this revision becomes valid.'))
    ->setRevisionable(TRUE);

  $definition_update_manager->updateFieldableEntityType($entity_type, $field_storage_definitions, $sandbox);

  return t('Shares have been converted to be revisionable.');
}
