<?php

namespace Drupal\shareholder_register\Controller;

use Symfony\Component\DependencyInjection\ContainerInterface;

use Drupal\Core\Url;
use Drupal\Component\Utility\Xss;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;

use Drupal\mail_template\MailTemplateServiceInterface;

use Drupal\shareholder_register\Entity\ShareholderInterface;

/**
 * Class ShareholderController.
 *
 *  Returns responses for Shareholder routes.
 */
class ShareholderController extends ControllerBase implements ContainerInjectionInterface {

  /**
   * Drupal\mail_template\MailTemplateServiceInterface definition.
   *
   * @var \Drupal\mail_template\MailTemplateServiceInterface
   */
  protected $mailTemplateDefault;

  /**
   * Constructs a new CreatePackagesForm object.
   */
  public function __construct(
    MailTemplateServiceInterface $mailTemplateDefault
  ) {
    $this->mailTemplateDefault = $mailTemplateDefault;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('mail_template.default')
    );
  }

  /**
   * Displays a Shareholder  revision.
   *
   * @param int $shareholder_revision
   *   The Shareholder  revision ID.
   *
   * @return array
   *   An array suitable for drupal_render().
   */
  public function revisionShow($shareholder_revision) {
    $shareholder = $this->entityTypeManager()->getStorage('shareholder')->loadRevision($shareholder_revision);
    $view_builder = $this->entityTypeManager()->getViewBuilder('shareholder');

    return $view_builder->view($shareholder);
  }

  /**
   * Page title callback for a Shareholder  revision.
   *
   * @param int $shareholder_revision
   *   The Shareholder  revision ID.
   *
   * @return string
   *   The page title.
   */
  public function revisionPageTitle($shareholder_revision) {
    $shareholder = $this->entityTypeManager()->getStorage('shareholder')->loadRevision($shareholder_revision);
    return $this->t('Revision of %title from %date', ['%title' => $shareholder->label(), '%date' => format_date($shareholder->getRevisionCreationTime())]);
  }

  /**
   * Generates an overview table of older revisions of a Shareholder .
   *
   * @param \Drupal\shareholder_register\Entity\ShareholderInterface $shareholder
   *   A Shareholder  object.
   *
   * @return array
   *   An array as expected by drupal_render().
   */
  public function revisionOverview(ShareholderInterface $shareholder) {
    $account = $this->currentUser();
    $langcode = $shareholder->language()->getId();
    $langname = $shareholder->language()->getName();
    $languages = $shareholder->getTranslationLanguages();
    $has_translations = (count($languages) > 1);
    $shareholder_storage = $this->entityTypeManager()->getStorage('shareholder');

    $build['#title'] = $has_translations ? $this->t('@langname revisions for %title', ['@langname' => $langname, '%title' => $shareholder->label()]) : $this->t('Revisions for %title', ['%title' => $shareholder->label()]);
    $header = [$this->t('Revision'), $this->t('Operations')];

    $revert_permission = (($account->hasPermission("revert all shareholder revisions") || $account->hasPermission('administer shareholder entities')));
    $delete_permission = (($account->hasPermission("delete all shareholder revisions") || $account->hasPermission('administer shareholder entities')));

    $rows = [];

    $vids = $shareholder_storage->revisionIds($shareholder);

    $latest_revision = TRUE;

    foreach (array_reverse($vids) as $vid) {
      /** @var \Drupal\shareholder_register\ShareholderInterface $revision */
      $revision = $shareholder_storage->loadRevision($vid);
      // Only show revisions that are affected by the language that is being
      // displayed.
      if ($revision->hasTranslation($langcode) && $revision->getTranslation($langcode)->isRevisionTranslationAffected()) {
        $username = [
          '#theme' => 'username',
          '#account' => $revision->getRevisionUser(),
        ];

        // Use revision link to link to revisions that are not active.
        $date = \Drupal::service('date.formatter')->format($revision->getRevisionCreationTime(), 'short');
        if ($vid != $shareholder->getRevisionId()) {
          $link = $this->l($date, new Url('entity.shareholder.revision', ['shareholder' => $shareholder->id(), 'shareholder_revision' => $vid]));
        }
        else {
          $link = $shareholder->link($date);
        }

        $row = [];
        $column = [
          'data' => [
            '#type' => 'inline_template',
            '#template' => '{% trans %}{{ date }} by {{ username }}{% endtrans %}{% if message %}<p class="revision-log">{{ message }}</p>{% endif %}',
            '#context' => [
              'date' => $link,
              'username' => \Drupal::service('renderer')->renderPlain($username),
              'message' => ['#markup' => $revision->getRevisionLogMessage(), '#allowed_tags' => Xss::getHtmlTagList()],
            ],
          ],
        ];
        $row[] = $column;

        if ($latest_revision) {
          $row[] = [
            'data' => [
              '#prefix' => '<em>',
              '#markup' => $this->t('Current revision'),
              '#suffix' => '</em>',
            ],
          ];
          foreach ($row as &$current) {
            $current['class'] = ['revision-current'];
          }
          $latest_revision = FALSE;
        }
        else {
          $links = [];
          if ($revert_permission) {
            $links['revert'] = [
              'title' => $this->t('Revert'),
              'url' => Url::fromRoute('entity.shareholder.revision_revert', ['shareholder' => $shareholder->id(), 'shareholder_revision' => $vid]),
            ];
          }

          /*
          if ($delete_permission) {
            $links['delete'] = [
              'title' => $this->t('Delete'),
              'url' => Url::fromRoute('entity.shareholder.revision_delete', ['shareholder' => $shareholder->id(), 'shareholder_revision' => $vid]),
            ];
          }
          */
          $row[] = [
            'data' => [
              '#type' => 'operations',
              '#links' => $links,
            ],
          ];
        }

        $rows[] = $row;
      }
    }

    $build['shareholder_revisions_table'] = [
      '#theme' => 'table',
      '#rows' => $rows,
      '#header' => $header,
    ];

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function downloadCertificate(ShareholderInterface $shareholder) {
    if ($shareholder->getState() != 'valid') {
      throw new \Exception('you cannot generate a certificate when the shareholder is not in state valid!');
    }

    $context = [
      'shareholder' => $shareholder,
      'langcode' => $shareholder->getPreferredLangcode(),
    ];

    $file = $this->mailTemplateDefault->pdfForRenderArrays(
      [$this->mailTemplateDefault->renderArrayForTemplate('shareholder_extract', $context)],
      $this->mailTemplateDefault->pdfOptionsForTemplate('shareholder_extract', $context));

    return $this->mailTemplateDefault->downloadForFile(
      $file, $filename = 'certificate.pdf');
  }
}
