<?php

namespace Drupal\shareholder_register\Element;

use Drupal\Component\Utility\Html;
use Drupal\Core\Render\Element\FormElement;
use Drupal\Core\Form\FormStateInterface;

use Drupal\shareholder_register\Plugin\Field\FieldType\AddressFieldType;

/**
 * Provides an address field form element.
 *
 * Usage example:
 * @code
 * $form['address'] = [
 *   '#type' => 'address_field',
 *   '#default_value' => [
 *     'given_name' => 'John',
 *     'family_name' => 'Smith',
 *     'organization' => 'Google Inc.',
 *     'address_line1' => '1098 Alta Ave',
 *     'postal_code' => '94043',
 *     'locality' => 'Mountain View',
 *     'administrative_area' => 'CA',
 *     'country_code' => 'US',
 *     'langcode' => 'en',
 *   ],
 *   '#available_countries' => ['DE', 'FR'],
 * ];
 * @endcode
 *
 * @FormElement("address_field")
 */
class AddressField extends FormElement {

  /**
   * {@inheritdoc}
   */
  public function getInfo() {
    $class = get_class($this);
    return [
      // List of country codes. If empty, all countries will be available.
      '#available_countries' => [],
      // List of AddressField constants. If empty, all fields will be used.
      '#used_fields' => [],

      '#input' => TRUE,
      '#multiple' => FALSE,
      '#default_value' => NULL,
      '#process' => [
        [$class, 'processAddress'],
        // [$class, 'processGroup'],
      ],
      '#pre_render' => [
        // [$class, 'groupElements'],
        // [$class, 'preRenderGroup'],
      ],
      '#after_build' => [
        // [$class, 'clearValues'],
      ],
      '#attached' => [
        'library' => ['shareholder_register/admin'],
      ],
      '#theme_wrappers' => ['container'],
    ];
  }

  /**
   * Processes the address form element.
   *
   * @param array $element
   *   The form element to process.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param array $complete_form
   *   The complete form structure.
   *
   * @return array
   *   The processed element.
   *
   * @throws \InvalidArgumentException
   *   Thrown when #used_fields is malformed.
   */
  public static function processAddress(array &$element, FormStateInterface $form_state, array &$complete_form) {

    if (isset($element['#used_fields']) && !is_array($element['#used_fields'])) {
      throw new \InvalidArgumentException('The #used_fields property must be an array.');
    }
    $id_prefix = implode('-', $element['#parents']);
    $wrapper_id = Html::getUniqueId($id_prefix . '-ajax-wrapper');
    // The #value has the new values on #ajax, the #default_value otherwise.
    $value = $element['#value'];

    $element = [
      '#tree' => TRUE,
      '#prefix' => '<div id="' . $wrapper_id . '">',
      '#suffix' => '</div>',
      // Pass the id along to other methods.
      '#wrapper_id' => $wrapper_id,
    ] + $element;

    $properties = AddressFieldType::getDefinition();
    $format = AddressFieldType::getFormat($element['#address_options']);
    foreach ($format as $line_index => $format_line) {

      $element['container'.$line_index] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['address-container-inline', 'shareholder-flexbox'],
        ],
      ];

      foreach ($format_line as $property_dict) {
        $property = $property_dict['name'];
        $definition = $properties[$property];

        $element['container'.$line_index][$property] = [
          '#type' => isset($property_dict['type'])?$property_dict['type']:'textfield',
          '#title' => $definition['label'],
          '#default_value' => isset($value[$property]) ? $value[$property] : '',
          '#required' => isset($property_dict['required'])?$property_dict['required']:FALSE,
          '#access' => isset($property_dict['access'])?$property_dict['access']:TRUE,

          //'#size' => $definition['size'],
          //'#attributes' => [
          // 'class' => [$class],
          // 'autocomplete' => FieldHelper::getAutocompleteAttribute($field),
          //],

          '#prefix' => '<div class="shareholder-flex shareholder-flex--' . $property_dict['flex'] . '"><div class="shareholder-flex--container">' . $element['#prefix'],
          '#suffix' => $element['#suffix'] . '</div></div>',
        ];

        if (isset($property_dict['type']) && $property_dict['type'] == 'select' && isset($property_dict['options'])) {
          $element['container'.$line_index][$property]['#options'] = $property_dict['options'];
        }

        if (isset($property_dict['default']) && $property_dict['default'] && !$element['container'.$line_index][$property]['#default_value']) {
          $element['container'.$line_index][$property]['#default_value'] = $property_dict['default'];
        }

      }
    }
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public static function valueCallback(&$element, $input, FormStateInterface $form_state) {
    $r = parent::valueCallback($element, $input, $form_state);

    $properties = AddressFieldType::getDefinition();
    $format = AddressFieldType::getFormat();
    foreach ($format as $line_index => $format_line) {
      foreach ($properties as $property => $property_definition) {
        if (isset($input["container{$line_index}"][$property])) {
          $r[$property] = $input["container{$line_index}"][$property];
        }
      }
    }
    return $r;
  }

}
