<?php

namespace Drupal\shareholder_register\Element;

use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\NestedArray;

use Drupal\Core\Render\Element\FormElement;
use Drupal\Core\Form\FormStateInterface;

use Drupal\shareholder_register\Entity\Shareholder;
use Drupal\shareholder_register\Plugin\Field\FieldType\AddressFieldType;

/**
 * Provides a shareholder shares at date field form element.
 *
 * Usage example:
 * @code
 * $form['shareholder_shares_at_date'] = [
 *   '#type' => 'shareholder_shares_at_date',
 * ];
 * @endcode
 *
 * @FormElement("shareholder_shares_at_date")
 */
class ShareholderSharesAtDate extends FormElement {

  /**
   * {@inheritdoc}
   */
  public function getInfo() {
    $class = get_class($this);
    return [
      '#input' => TRUE,
      '#multiple' => FALSE,
      '#default_value' => NULL,
      '#process' => [
        [$class, 'processShareholderSharesAtDate'],
      ],
      '#element_validate' => [
        [$class, 'validateShareholderSharesAtDate']
      ],
      '#attached' => [
        'library' => ['shareholder_register/admin'],
      ],
      '#theme_wrappers' => ['container'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function validateShareholderSharesAtDate(array &$element, FormStateInterface $form_state, array &$complete_form) {
    $shareholder_id = $form_state->getValue(array_merge($element['#parents'], ['shareholder']));
    if (!$form_state->has('shareholder_id') || $form_state->get('shareholder_id') != $shareholder_id) {
      $form_state->set('shareholder_id', $shareholder_id);
      $form_state->setRebuild();
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function processShareholderSharesAtDate(&$element, FormStateInterface $form_state, &$complete_form) {
    $element['#tree'] = TRUE;

    $element['date'] = [
      '#type' => 'date',
      '#title' => t('Date'),
      '#weight' => '10',
      '#ajax' => [
        'callback' => [get_called_class(), 'updateShares'],
        'event' => 'change',
        'wrapper' => 'convert-form-shares',
      ],
      '#default_value' => $element['#default_value']['date'],
    ];

    $element['shareholder'] = [
      '#type' => 'entity_autocomplete',
      '#title' => t('Shareholder'),
      '#weight' => '20',
      '#required' => TRUE,
      '#target_type' => 'shareholder',
      '#ajax' => [
        'callback' => [get_called_class(), 'updateShares'],
        'event' => 'change',
        'wrapper' => 'convert-form-shares',
      ],
    ];

    if (isset($element['#default_value']['shareholder']) &&
      $element['#default_value']['shareholder']) {
      $element['shareholder']['#default_value'] = Shareholder::load($element['#default_value']['shareholder']);
      // $element['shareholder']['#disabled'] = TRUE;
      if (!$form_state->has('shareholder_id')) {
        $form_state->set('shareholder_id', $element['#default_value']['shareholder']);
      }
    }

    $options = [];
    if ($form_state->has('shareholder_id')) {
      $date = $form_state->getValue(array_merge($element['#parents'], ['date']));
      if ($s = Shareholder::load($form_state->get('shareholder_id'))) {
        $shares = $s->getShareRevisionsAtDate($date);
        foreach ($shares as $share) {
          $options[$share->id()] = $share->getHashLabel();
        }
      }
    }

    $element['shares'] = [
      '#type' => 'select',
      '#title' => t('Shares'),
      '#weight' => '30',
      '#required' => TRUE,
      '#target_type' => 'share',
      '#multiple' => TRUE,
      '#size' => 10,
      '#options' => $options,
      '#prefix' => '<div id="convert-form-shares">',
      '#suffix' => '</div>',
      '#default_value' => !empty($form_state->getValue(array_merge($element['#parents'], ['shares']))) ? $form_state->getValue(array_merge($element['#parents'], ['shares'])) : '',
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public static function updateShares(array $form, FormStateInterface $form_state) {
    $triggering_element = $form_state->getTriggeringElement();
    $parents = $triggering_element['#array_parents'];
    $shares_element_key = array_slice($parents, 0, -1);
    $shares_element_key[] = 'shares';
    $shares_element = NestedArray::getValue($form, $shares_element_key);
    return $shares_element;
  }
}
