<?php

namespace Drupal\shareholder_register\Entity;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Entity\RevisionableContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\user\UserInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Defines the Share entity.
 *
 * @ingroup shareholder_register
 *
 * @ContentEntityType(
 *   id = "share",
 *   label = @Translation("Share"),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\shareholder_register\ShareListBuilder",
 *     "views_data" = "Drupal\shareholder_register\Entity\ShareViewsData",
 *     "form" = {
 *       "default" = "Drupal\shareholder_register\Form\ShareForm",
 *       "add" = "Drupal\shareholder_register\Form\ShareForm",
 *       "edit" = "Drupal\shareholder_register\Form\ShareForm",
 *       "delete" = "Drupal\shareholder_register\Form\ShareDeleteForm",
 *     },
 *     "access" = "Drupal\shareholder_register\ShareAccessControlHandler",
 *     "route_provider" = {
 *       "html" = "Drupal\shareholder_register\ShareHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "share",
 *   revision_table = "share_revision",
 *   revision_data_table = "share_field_revision",
 *   revision_metadata_keys = {
 *     "revision_default" = "revision_default",
 *     "revision_user" = "revision_user",
 *     "revision_created" = "revision_created",
 *     "revision_log_message" = "revision_log_message",
 *     "revision_date" = "revision_date",
 *   },
 *   admin_permission = "administer share entities",
 *   entity_keys = {
 *     "id" = "id",
 *     "revision" = "vid",
 *     "label" = "name",
 *     "uuid" = "uuid",
 *     "uid" = "user_id",
 *     "revision" = "vid"
 *   },
 *   links = {
 *     "canonical" = "/admin/shareholder_register/share/{share}",
 *     "add-form" = "/admin/shareholder_register/share/add",
 *     "edit-form" = "/admin/shareholder_register/share/{share}/edit",
 *     "delete-form" = "/admin/shareholder_register/share/{share}/delete",
 *     "collection" = "/admin/shareholder_register/share_type/share",
 *   },
 *   field_ui_base_route = "share.settings"
 * )
 */
class Share extends RevisionableContentEntityBase implements ShareInterface {

  use EntityChangedTrait;
  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage_controller, array &$values) {
    parent::preCreate($storage_controller, $values);

    $values += [
      'user_id' => \Drupal::currentUser()->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage) {
    parent::preSave($storage);

    if ($this->id()) {
      $original = $storage->loadUnchanged($this->id());
      if ($original && $original->get('state')->value == 'valid') {
        foreach ($this->getHashFields() as $f) {
          if (in_array($this->getFieldDefinition($f)->getType(), [
                'string',
                'datetime',
                'decimal',
                'integer'
          ])) {
            if ($original->get($f)->value != $this->get($f)->value) {
              throw new \Exception('you cannot modify base fields when the share is in state valid!');
            }
          } elseif (in_array($this->getFieldDefinition($f)->getType(), [
                'entity_reference'
          ])) {
            if ($original->get($f)->target_id != $this->get($f)->target_id) {
              throw new \Exception('you cannot modify base fields when the share is in state valid!');
            }
          }
        }
      }
    }

    $holder = $this->getShareholder();
    if ($holder && $this->get('shareholder_id')->target_id != $holder->id()) {
      $this->set('shareholder_id', $holder->id());
    }
    elseif ($holder === NULL && $this->get('shareholder_id')->target_id) {
      $this->set('shareholder_id', $holder);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function delete() {
    if ($this->getName()) {
      drupal_set_message($this->t("You cannot delete an issued share!"), 'error');
    } else {
      return parent::delete();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getHashFields($context=NULL) {
    // REVIEW: keep config in third party settings of properties
    if ($this->get('state')->value == 'valid') {
      return [
        'share_type_id',
      ];
    }
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function label() {
    if ($this->getIssuingTransaction()) {
      return $this->t("[#@number] @share_type (@date)", [
        '@date' => $this->getIssuingTransaction()->getDate(),
        '@number' => $this->getName(),
        '@share_type' => $this->getShareType()->label(),
      ]);
    }
    else {
      return $this->t("[draft] @share_type", [
        '@share_type' => $this->getShareType()->label(),
      ]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getHashLabel() {
    if ($this->getIssuingTransaction()) {
      $service = \Drupal::service('shareholder_register.default');
      return $this->t("[#@number] @hash (@date)", [
        '@date' => $this->getIssuingTransaction()->getDate(),
        '@number' => $this->getName(),
        '@hash' => $service->getShareHashText(
          $service->getShareHash($this)),
      ]);
    }
    else {
      return $this->t("[draft] @share_type", [
        '@share_type' => $this->getShareType()->label(),
      ]);
    }
  }


  /**
   * {@inheritdoc}
   */
  public function getName() {
    return $this->get('name')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setName($name) {
    $this->set('name', $name);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getState() {
    return $this->get('state')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setState($state) {
    return $this->set('state', $state);
  }

  /**
   * {@inheritdoc}
   */
  public function getShareType() {
    return $this->get('share_type_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getTransactions() {
    return $this->get('share_transaction_ids')->referencedEntities();
  }

  /**
   * {@inheritdoc}
   */
  public function getValidTransactions() {
    return array_filter(
      $this->getTransactions(),
      function ($t) {
        return $t->getState() == 'valid';
      }
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getIssuingTransaction() {
    return $this->get('share_transaction_ids')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getIssueDate() {
    return $this->get('share_transaction_ids')->entity->getDate();
  }

  /**
   * {@inheritdoc}
   */
  public function getPaymentDate() {
    return $this->get('share_transaction_ids')->entity->getPaymentDate();
  }

  /**
   * {@inheritdoc}
   */
  public function getShareholder() {
    # FIXME: at date
    $transactions = $this->getValidTransactions();
    $last_transaction = end($transactions);
    if ($last_transaction &&
      $last_transaction->getShareGroupTypeBaseType() !== ShareTransactionGroupType::BASE_REDEMPTION) {
      return $last_transaction->getShareholder();
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getIssuePremium() {
    return $this->get('issue_premium')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getShareValue() {
    return $this->get('share_value')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime($timestamp) {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwner() {
    return $this->get('user_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwnerId() {
    return $this->get('user_id')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwnerId($uid) {
    $this->set('user_id', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwner(UserInterface $account) {
    $this->set('user_id', $account->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getSharePriceComponents() {
    $price_items = [];

    if ($this->get('share_value')->value) {
      $price_items['par_value'] = [
        'label' => $this->t('Par Value'),
        'name' => 'par_value',
        'amount' => $this->get('share_value')->value,
      ];
    }

    if ($this->get('issue_premium')->value) {
      $price_items['issue_premium'] = [
        'label' => $this->t('Issue Premium'),
        'name' => 'issue_premium',
        'amount' => $this->get('issue_premium')->value,
      ];
    }

    $price_items = array_merge(
      $price_items,
      \Drupal::moduleHandler()->invokeAll('share_price', [$this])
    );

    \Drupal::moduleHandler()->alter('share_price', $this, $price_items);
    return $price_items;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['user_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Authored by'))
      ->setDescription(t('The user ID of author of the Share entity.'))
      ->setRevisionable(TRUE)
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Name'))
      ->setDescription(t('The name of the Share entity.'))
      ->setSettings([
        'max_length' => 50,
        'text_processing' => 0,
        'wkf-editable' => ['state' => []],
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    $fields['shareholder_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Current Holder'))
      ->setDescription(t('The Current Shareholder.'))
      ->setReadonly(TRUE)
      ->setRevisionable(FALSE)
      ->setSettings([
        'target_type' => 'shareholder',
        'default_value' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'weight' => 2,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['share_type_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Share Type'))
      ->setDescription(t('The Share Type.'))
      ->setRevisionable(TRUE)
      ->setSettings(array(
        'target_type' => 'share_type',
        'default_value' => 0,
      ))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'author',
        'weight' => 3,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 3,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['share_value'] = BaseFieldDefinition::create('decimal')
      ->setLabel(t('Share Value'))
      ->setDescription(t('The Value (nominal or fractional) of this Share.'))
      ->setSettings([
        'wkf-editable' => ['state' => []],
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'number_decimal',
        'weight' => 6,
      ])
      ->setDisplayOptions('form', [
        'type' => 'number',
        'weight' => 6,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['issue_premium'] = BaseFieldDefinition::create('decimal')
      ->setLabel(t('Issue Premium'))
      ->setDescription(t('The Issue Premium for this Share.'))
      ->setSettings([
        'wkf-editable' => ['state' => []],
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'number_decimal',
        'weight' => 6,
      ])
      ->setDisplayOptions('form', [
        'type' => 'number',
        'weight' => 6,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['exit_disbursement'] = BaseFieldDefinition::create('decimal')
      ->setLabel(t('Exit disbursement'))
      ->setDescription(t('The Exit Disbursement for this Share.'))
      ->setSettings([
        'wkf-editable' => ['state' => []],
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'number_decimal',
        'weight' => 7,
      ])
      ->setDisplayOptions('form', [
        'type' => 'number',
        'weight' => 7,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['share_transaction_ids'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Share Transactions'))
      ->setDescription(t('The Share Transactions.'))
      ->setCardinality(FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED)
      ->setRevisionable(FALSE)
      ->setSettings(array(
        'target_type' => 'share_transaction',
        'default_value' => 0,
        'wkf-editable' => ['state' => []],
      ))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'share_transactions_formatter',
        'weight' => 3,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 3,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['state'] = BaseFieldDefinition::create('string')
      ->setLabel(t('State'))
      ->setDescription(t('State of the Share.'))
      ->setRevisionable(TRUE)
      ->setDefaultValue('draft');

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the entity was last edited.'));

    $fields[static::getRevisionMetadataKey($entity_type, 'revision_date')] = BaseFieldDefinition::create('datetime')
      ->setLabel(t('Revision date'))
      ->setDescription(t('The date at which this revision becomes valid.'))
      ->setRevisionable(TRUE);

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function getShareHash($context=NULL) {
    $hash_keys = [
    ];
    foreach ($this->getHashFields($context) as $f) {
      if (in_array($this->getFieldDefinition($f)->getType(), [
            'string',
            'datetime',
            'decimal',
            'integer'
      ])) {
        $hash_keys[] = $this->get($f)->value;
      } elseif (in_array($this->getFieldDefinition($f)->getType(), [
            'entity_reference'
      ])) {
        $hash_keys[] = $this->get($f)->target_id;
      }
    }

    \Drupal::moduleHandler()->alter('share_hash', $hash_keys, $context);
    return implode('-', $hash_keys);
  }


  /**
   * {@inheritdoc}
   */
  public function actionIssue() {
    if (!$this->getIssuingTransaction() || $this->getState() !== 'draft') {
      // FIXME: Set error
      return;
    }
    if (!$this->getName()) {
      $sid = "share_number";
      $previous_number = \Drupal::state()->get($sid);
      if ($previous_number) {
        $next_number = $previous_number + 1 ;
      }
      else {
        $connection = \Drupal::database();
        $result = $connection->query("select max(convert(coalesce(name, 0), SIGNED INTEGER)) + 1 as maxid from {share}");
        $next_number = $result->fetchObject()->maxid;
        if (!$next_number) {
          $next_number = 1;
        }
      }
      $this->setName($next_number);
      \Drupal::state()->set($sid, $next_number);
    }

    $transaction = $this->getIssuingTransaction();
    $this->share_type_id->target_id = $transaction->getShareType()->id();
    $this->share_value->value = $transaction->getShareValue();
    $this->issue_premium->value = $transaction->getIssuePremium();

    $this->state->value = 'issued';
    $this->save();
  }

  /**
   * {@inheritdoc}
   */
  public function actionRedeem() {
    $this->state->value = 'redeemed';
    $this->save();
  }
}
