<?php

namespace Drupal\shareholder_register\Entity;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

use Drupal\user\UserInterface;

/**
 * Defines the Share issue entity.
 *
 * @ingroup shareholder_register
 *
 * @ContentEntityType(
 *   id = "share_issue",
 *   label = @Translation("Share issue"),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\shareholder_register\ShareIssueListBuilder",
 *     "views_data" = "Drupal\shareholder_register\Entity\ShareIssueViewsData",
 *
 *     "form" = {
 *       "default" = "Drupal\shareholder_register\Form\ShareIssueForm",
 *       "add" = "Drupal\shareholder_register\Form\ShareIssueForm",
 *       "edit" = "Drupal\shareholder_register\Form\ShareIssueForm",
 *       "delete" = "Drupal\shareholder_register\Form\ShareIssueDeleteForm",
 *     },
 *     "access" = "Drupal\shareholder_register\ShareIssueAccessControlHandler",
 *     "route_provider" = {
 *       "html" = "Drupal\shareholder_register\ShareIssueHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "share_issue",
 *   admin_permission = "administer share issue entities",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "name",
 *     "uuid" = "uuid",
 *     "uid" = "user_id",
 *     "langcode" = "langcode"
 *   },
 *   links = {
 *     "canonical" = "/admin/shareholder_register/share_type/share_issue/{share_issue}",
 *     "add-form" = "/admin/shareholder_register/share_type/share_issue/add",
 *     "edit-form" = "/admin/shareholder_register/share_type/share_issue/{share_issue}/edit",
 *     "delete-form" = "/admin/shareholder_register/share_type/share_issue/{share_issue}/delete",
 *     "collection" = "/admin/shareholder_register/share_type/share_issue",
 *   },
 *   field_ui_base_route = "share_issue.settings"
 * )
 */
class ShareIssue extends ContentEntityBase implements ShareIssueInterface {

  use EntityChangedTrait;
  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage_controller, array &$values) {
    parent::preCreate($storage_controller, $values);
    $values += [
      'user_id' => \Drupal::currentUser()->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getName() {
    return $this->get('name')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setName($name) {
    $this->set('name', $name);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime($timestamp) {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwner() {
    return $this->get('user_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwnerId() {
    return $this->get('user_id')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwnerId($uid) {
    $this->set('user_id', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwner(UserInterface $account) {
    $this->set('user_id', $account->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getIssuePremium() {
    return $this->get('issue_premium')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getParValue() {
    return $this->get('share_type_id')->entity->getParValue();
  }

  /**
   * {@inheritdoc}
   */
  public function getShareType() {
    return $this->get('share_type_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getSharePrice() {
    return array_sum(
      array_map(
        function ($pc) {
          return $pc['amount'];
        },
        $this->getSharePriceComponents()
      ));
  }

  /**
   * {@inheritdoc}
   */
  public function getSharePriceComponents() {
    $price_items = [
      'par_value' => [
        'label' => $this->t('Par Value'),
        'name' => 'par_value',
        'amount' => $this->get('share_type_id')->entity->get('par_value')->value,
      ],
      'issue_premium' => [
        'label' => $this->t('Issue Premium'),
        'name' => 'issue_premium',
        'amount' => $this->get('issue_premium')->value,
      ],
    ];
    $price_items = array_merge(
      $price_items,
      \Drupal::moduleHandler()->invokeAll('share_issue_price', [$this])
    );

    \Drupal::moduleHandler()->alter('share_issue_price', $this, $price_items);

    return $price_items;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['user_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Authored by'))
      ->setDescription(t('The user ID of author of the Share issue entity.'))
      ->setRevisionable(TRUE)
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setTranslatable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'author',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 5,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Name'))
      ->setDescription(t('The name of the Share issue entity.'))
      ->setSettings([
        'max_length' => 50,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    $fields['share_type_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Share Type'))
      ->setDescription(t('The Share Type.'))
      ->setSettings(array(
        'target_type' => 'share_type',
        'default_value' => 0,
        'wkf-editable' => ['state' => ['draft']],
      ))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'author',
        'weight' => 3,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 3,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    $fields['issue_premium'] = BaseFieldDefinition::create('decimal')
      ->setLabel(t('Issue Premium'))
      ->setDescription(t('The Issue Premium per Share for this Transaction.'))
      ->setSettings(array(
        'wkf-editable' => ['state' => ['draft']],
      ))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'number_decimal',
        'weight' => 6,
      ])
      ->setDisplayOptions('form', [
        'type' => 'number',
        'weight' => 6,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['state'] = BaseFieldDefinition::create('string')
      ->setLabel(t('State'))
      ->setDescription(t('State of the Shareholder.'))
      ->setRevisionable(TRUE)
      ->setDefaultValue('draft');

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the entity was last edited.'));

    return $fields;
  }

}
