<?php

namespace Drupal\shareholder_register\Entity;

use Drupal\Core\StringTranslation\StringTranslationTrait;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\user\UserInterface;

use Drupal\shareholder_register\Exception\ShareholderRegisterMissingSharesException;
use Drupal\shareholder_register\Exception\ShareTransactionNegativeShareCountException;
use Drupal\shareholder_register\Exception\ShareTransactionNegativeShareCountAtDateException;
use Drupal\shareholder_register\Event\ShareTransactionEvent;

/**
 * Defines the Share transaction entity.
 *
 * @ingroup shareholder_register
 *
 * @ContentEntityType(
 *   id = "share_transaction",
 *   label = @Translation("Share transaction"),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\shareholder_register\ShareTransactionListBuilder",
 *     "views_data" = "Drupal\shareholder_register\Entity\ShareTransactionViewsData",
 *
 *     "form" = {
 *       "default" = "Drupal\shareholder_register\Form\ShareTransactionForm",
 *       "issue" = "Drupal\shareholder_register\Form\IssueShareTransactionForm",
 *       "repurchase" = "Drupal\shareholder_register\Form\RepurchaseShareTransactionForm",
 *       "edit" = "Drupal\shareholder_register\Form\ShareTransactionForm",
 *       "delete" = "Drupal\shareholder_register\Form\ShareTransactionDeleteForm",
 *     },
 *     "access" = "Drupal\shareholder_register\ShareTransactionAccessControlHandler",
 *     "route_provider" = {
 *       "html" = "Drupal\shareholder_register\ShareTransactionHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "share_transaction",
 *   admin_permission = "administer share transaction entities",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "name",
 *     "uuid" = "uuid",
 *     "uid" = "user_id",
 *   },
 *   links = {
 *     "canonical" = "/admin/shareholder_register/share_transaction/{share_transaction}",
 *     "edit-form" = "/admin/shareholder_register/share_transaction/{share_transaction}/edit",
 *     "delete-form" = "/admin/shareholder_register/share_transaction/{share_transaction}/delete",
 *     "collection" = "/admin/shareholder_register/share_transaction",
 *     "validate" = "/admin/shareholder_register/share_transaction/{share_transaction}/validate",
 *     "refuse" = "/admin/shareholder_register/share_transaction/{share_transaction}/refuse",
 *     "extract" = "/admin/shareholder_register/share_transaction/{share_transaction}/downloadExtract",
 *     "request" = "/admin/shareholder_register/share_transaction/{share_transaction}/downloadRequest",
 *   },
 *   field_ui_base_route = "share_transaction.settings",
 *   constraints = {
 *     "share_issue_constraint" = {}
 *   }
 * )
 */
class ShareTransaction extends ContentEntityBase implements ShareTransactionInterface {

  use EntityChangedTrait;
  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage_controller, array &$values) {
    parent::preCreate($storage_controller, $values);

    $values += [
      'user_id' => \Drupal::currentUser()->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function postCreate(EntityStorageInterface $storage_controller) {

  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage) {

    if (!$this->getShareType() && $this->getShareIssue()) {
      $this->get('share_type_id')->target_id = $this->getShareIssue()->getShareType()->id();
      $this->get('issue_premium')->value = $this->getShareIssue()->getIssuePremium();
    }

    // Also test for share type with nominal value.
    if ($this->getShareType()) {
      $this->get('share_value')->value = $this->getShareType()->getParValue();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function delete() {
    if ($this->getName()) {
      drupal_set_message("You cannot delete a validated transaction!", 'error');
    }
    else {
      return parent::delete();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function label() {
    if ($this->getQuantity() > 0) {
      return $this->t("[@number] Acquisition of @shares shares for @shareholder on @date.", [
        '@number' => $this->getName(),
        '@shares' => $this->getQuantity(),
        '@shareholder' => $this->getShareholder()->label(),
        '@date' => $this->getDate(),
      ]);
    }
    else {
      return $this->t("[@number] Disposal from @shares shares for @shareholder on @date.", [
        '@number' => $this->getName(),
        '@shares' => $this->getQuantity(),
        '@shareholder' => $this->getShareholder()->label(),
        '@date' => $this->getDate(),
      ]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getName() {
    return $this->get('name')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setName($name) {
    $this->set('name', $name);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getQuantity() {
    return $this->get('quantity')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setQuantity($quantity) {
    return $this->set('quantity', $quantity);
  }

  /**
   * {@inheritdoc}
   */
  public function getIssuePremium() {
    return $this->get('issue_premium')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getParValue() {
    return $this->get('share_type_id')->entity->getParValue();
  }

  /**
   * {@inheritdoc}
   */
  public function getShareValue() {
    return $this->get('share_value')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getTotalValue() {
    return bcmul(
      $this->get('quantity')->value,
      bcadd(
        $this->getIssuePremium(),
        $this->getParValue(),
        2),
      2);
  }

  /**
   * {@inheritdoc}
   */
  public function getDate() {
    return $this->get('date')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setDate($date) {
    return $this->set('date', $date);
  }

  /**
   * {@inheritdoc}
   */
  public function getPaymentDate() {
    return $this->get('payment_date')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setPaymentDate($date) {
    return $this->set('payment_date', $date);
  }

  /**
   * {@inheritdoc}
   */
  public function getTransactionDate($use_payment_date = FALSE) {
    if ($use_payment_date && $this->get('payment_date')->value) {
      return $this->get('payment_date')->value;
    }
    else {
      return $this->get('date')->value;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getNotes() {
    return $this->get('notes')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setNotes($notes) {
    return $this->set('notes', $notes);
  }

  /**
   * {@inheritdoc}
   */
  public function getState() {
    return $this->get('state')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setState($state) {
    return $this->set('state', $state);
  }

  /**
   * {@inheritdoc}
   */
  public function getShareType() {
    return $this->get('share_type_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getShareIssue() {
    return $this->get('share_issue_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getShareholder() {
    return $this->get('shareholder_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getShares() {
    return $this->get('shares')->referencedEntities();
  }

  /**
   * {@inheritdoc}
   */
  public function getShareIds() {
    return \Drupal::entityQuery('share')
      ->condition('share_transaction_ids', $this->id())
      ->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function getTransactionGroup() {
    return $this->get('transaction_group')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getShareGroupTypeBaseType() {
    if ($this->get('transaction_group')->entity) {
      return $this->getTransactionGroup()->getShareGroupTypeBaseType();
    }
    else {
      if ($this->getQuantity() > 0) {
        return ShareTransactionGroupType::BASE_ISSUE;
      }
      else {
        return ShareTransactionGroupType::BASE_REDEMPTION;
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime($timestamp) {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwner() {
    return $this->get('user_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwnerId() {
    return $this->get('user_id')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwnerId($uid) {
    $this->set('user_id', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwner(UserInterface $account) {
    $this->set('user_id', $account->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormattedShareNames() {
    return \Drupal::service('shareholder_register.formatter')->sharesToRanges(
      $this->get('shares')->getReferencedEntities());
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['user_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Authored by'))
      ->setDescription(t('The user ID of author of the Share transaction entity.'))
      ->setRevisionable(TRUE)
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setTranslatable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'author',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Transaction Number'))
      ->setDescription(t('The Number of the Share transaction entity.'))
      ->setSettings([
        'max_length' => 50,
        'text_processing' => 0,
        'wkf-editable' => ['state' => []],
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 1,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['transaction_group'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Transaction Group'))
      ->setDescription(t('The Share Transaction Group.'))
      ->setSettings(array(
        'target_type' => 'share_transaction_group',
        'default_value' => 0,
        'wkf-editable' => ['state' => []],
      ))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'author',
        'weight' => 2,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 2,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['shareholder_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Shareholder'))
      ->setDescription(t('The Shareholder.'))
      ->setSettings(array(
        'target_type' => 'shareholder',
        'default_value' => 0,
        'wkf-editable' => ['state' => ['draft']],
      ))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'weight' => 2,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 2,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);


    $fields['share_type_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Share Type'))
      ->setDescription(t('The Share Type.'))
      ->setSettings(array(
        'target_type' => 'share_type',
        'default_value' => 0,
      ))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'author',
        'weight' => 3,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 3,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['share_issue_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Share Issue'))
      ->setDescription(t('The Share Issue.'))
      ->setSettings(array(
        'target_type' => 'share_issue',
        'default_value' => 0,
        'wkf-editable' => ['state' => ['draft']],
      ))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'author',
        'weight' => 3,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 3,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['shares'] = BaseFieldDefinition::create('share_transaction_shares')
      ->setLabel(t('Shares'))
      ->setComputed(TRUE)
      ->setSettings([
        'target_type' => 'share',
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'weight' => 15,
      ]);

    $fields['date'] = BaseFieldDefinition::create('datetime')
      ->setLabel(t('Transaction Date'))
      ->setDescription(t('The Date of the Share transaction entity.'))
      ->setSettings(array(
        'datetime_type' => 'date',
        'wkf-editable' => ['state' => ['draft']],
      ))
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['payment_date'] = BaseFieldDefinition::create('datetime')
      ->setLabel(t('Date of Payment'))
      ->setDescription(t('The Date the Share transaction was paid in full.'))
      ->setSettings(array(
        'datetime_type' => 'date',
        'wkf-editable' => ['state' => ['draft']],
      ))
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['quantity'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Quantity'))
      ->setDescription(t('The Number of Shares in this transaction.'))
      ->setSettings(array(
        'wkf-editable' => ['state' => ['draft']],
      ))
      ->setDefaultValue(1)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 5,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['share_value'] = BaseFieldDefinition::create('decimal')
      ->setLabel(t('Share Value'))
      ->setDescription(t('The Value (nominal or fractional) per Share for this Transaction.'))
      ->setSettings(array(
        'wkf-editable' => ['state' => ['draft']],
      ))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'number_decimal',
        'weight' => 6,
      ])
      ->setDisplayOptions('form', [
        'type' => 'number',
        'weight' => 6,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['issue_premium'] = BaseFieldDefinition::create('decimal')
      ->setLabel(t('Issue Premium'))
      ->setDescription(t('The Issue Premium per Share for this Transaction.'))
      ->setSettings(array(
        'wkf-editable' => ['state' => ['draft']],
      ))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'number_decimal',
        'weight' => 6,
      ])
      ->setDisplayOptions('form', [
        'type' => 'number',
        'weight' => 6,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['exit_disbursement'] = BaseFieldDefinition::create('decimal')
      ->setLabel(t('Exit disbursement'))
      ->setDescription(t('The Exit Disbursement per Share for this Transaction.'))
      ->setSettings(array(
        'wkf-editable' => ['state' => ['draft']],
      ))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'number_decimal',
        'weight' => 7,
      ])
      ->setDisplayOptions('form', [
        'type' => 'number',
        'weight' => 7,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['notes'] = BaseFieldDefinition::create('text_long')
      ->setLabel(t('Notes'))
      ->setDescription(t('Notes for this transaction'))
      ->setDisplayOptions('view', array(
        'type' => 'text_default',
        'weight' => 10,
      ))
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('form', array(
        'type' => 'text_textfield',
        'weight' => 10,
      ));

    $fields['state'] = BaseFieldDefinition::create('simple_workflow_state_field')
      ->setLabel(t('State'))
      ->setReadonly(TRUE)
      ->setDescription(t('State of the Share Transaction.'))
      ->setDefaultValue('draft');

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the entity was last edited.'));

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function preValidateTransaction($date) {
    if ($this->getShareholder()) {
      $total_at_date = count($this->getShareholder()->getShareIdsAtDate($date));
      $total = count($this->getShareholder()->getShareIdsAtDate());
    }
    else {
      $total_at_date = $total = 0;
    }

    if (($total + $this->getQuantity()) < 0) {
      throw new ShareTransactionNegativeShareCountException();
    }
    if (($total_at_date + $this->getQuantity()) < 0) {
      throw new ShareTransactionNegativeShareCountAtDateException($date);
    }

    \Drupal::moduleHandler()->alter('share_transaction_pre_validate', $this, $date, $result);
  }

  /**
   * {@inheritdoc}
   */
  public function attachSharesToTransaction($date, $context = []) {
    $return = [];

    $attached_ids = $this->getShareIds();
    if (abs($this->getQuantity()) == count($attached_ids)) {
      return $attached_ids;
    }

    if ($this->getQuantity() == 0) {
      throw new ShareholderRegisterMissingSharesException();
    }

    if ($this->getQuantity() < 0) {
      // Select shares for removal.
      if (isset($context['share_ids']) && count($context['share_ids'])) {
        if (count($context['share_ids']) != abs($this->getQuantity())) {
          throw new ShareholderRegisterMissingSharesException();
        }

        foreach (Share::loadMultiple($context['share_ids']) as $share) {
          $share->get('share_transaction_ids')->appendItem($this->id());
          $share->save();
          $return[] = $share->id();
        }
      }
      else {
        $share_ids = array_values($this->getShareholder()->getCurrentShareIdsAtDate($date));
        for ($i = 0; $i < abs($this->getQuantity()); $i++) {
          $share = Share::load($share_ids[$i]);
          $share->get('share_transaction_ids')->appendItem($this->id());
          $share->save();
          $return[] = $share->id();
        }
      }
    }
    else {
      if (isset($context['share_ids']) && count($context['share_ids'])) {
        if (count($context['share_ids']) != $this->getQuantity()) {
          throw new ShareholderRegisterMissingSharesException();
        }
        foreach (Share::loadMultiple($context['share_ids']) as $share) {
          $share->get('share_transaction_ids')->appendItem($this->id());
          $share->save();
        }
        $return = $context['share_ids'];
      }
      else {
        if ($this->getShareGroupTypeBaseType() != ShareTransactionGroupType::BASE_ISSUE) {
          throw new ShareholderRegisterMissingSharesException();
        }

        // Issuing new shares.

        $transaction_field_definitions = \Drupal::service('entity_field.manager')->getFieldDefinitions(
          'share_transaction', 'share_transaction');

        for ($i = 0; $i < $this->getQuantity(); $i++) {
          $share = Share::create([
            'share_transaction_ids' => [$this->id()],
            'share_type_id' => $this->getShareType()->id(),
          ]);

          // Copy field_x fields to share if they exist.
          foreach ($transaction_field_definitions as $k => $v) {
            if (substr($k, 0, strlen('field_')) === 'field_' &&
              $share->hasField($k)) {
              $share->set(
                $k,
                $this->get($k)->getValue()
              );
            }
          }

          \Drupal::moduleHandler()->alter('share_issue', $this, $share);
          $share->save();
          $return[] = $share->id();
        }
      }
    }
    return $return;
  }

  /**
   * {@inheritdoc}
   */
  public function actionValidate($date, $context = []) {
    // Create transaction group if missing.
    if ($this->get('transaction_group')->isEmpty()) {
      if ($this->getQuantity() > 0) {
        $transaction_group_type = 'issue';
      }
      else {
        $transaction_group_type = 'repurchase';
      }

      $group = ShareTransactionGroup::create([
        'type' => $transaction_group_type,
      ]);
      $group->save();
      $this->set('transaction_group', $group);
      $this->save();
    }

    $this->getTransactionGroup()->actionValidate($date, $context);
  }

  /**
   * {@inheritdoc}
   */
  public function actionRefuse($date, $notes) {
    if ($this->getName()) {
      drupal_set_message($this->t('Share transaction @name already validated!', ['@name' => $this->getName()]), 'error');
      return $this;
    }

    foreach ($this->shareholder_id->referencedEntities() as $shareholder) {
      $shareholder->actionRefuse($date);
    }

    $this->setNotes($notes);
    $this->setState('refused');
    $this->save();

    $event = new ShareTransactionEvent($this);
    $event_dispatcher = \Drupal::service('event_dispatcher');
    $event_dispatcher->dispatch(ShareTransactionEvent::REFUSED_EVENT_NAME, $event);

    return $this->getName();
  }

  /**
   * {@inheritdoc}
   */
  public function actionCancel() {
    if ($this->getName()) {
      drupal_set_message($this->t('Share transaction @name already validated!', ['@name' => $this->getName()]), 'error');
      return $this;
    }

    $this->setState('cancel');
    $this->save();

    $event = new ShareTransactionEvent($this);
    $event_dispatcher = \Drupal::service('event_dispatcher');
    $event_dispatcher->dispatch(ShareTransactionEvent::CANCELED_EVENT_NAME, $event);

    return $this->getName();
  }

}
