<?php

namespace Drupal\shareholder_register\Entity;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\user\UserInterface;

use Drupal\shareholder_register\Exception\InvalidShareTransactionGroupException;
use Drupal\shareholder_register\Exception\ShareholderRegisterInvalidSharesException;

use Drupal\shareholder_register\Event\ShareTransactionEvent;
use Drupal\shareholder_register\Event\ShareTransactionGroupEvent;

/**
 * Defines the Share Transaction Group entity.
 *
 * @ingroup shareholder_register
 *
 * @ContentEntityType(
 *   id = "share_transaction_group",
 *   label = @Translation("Share Transaction Group"),
 *   bundle_label = @Translation("Share Transaction Group type"),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\shareholder_register\ShareTransactionGroupListBuilder",
 *     "views_data" = "Drupal\shareholder_register\Entity\ShareTransactionGroupViewsData",
 *
 *     "form" = {
 *       "default" = "Drupal\shareholder_register\Form\ShareTransactionGroupForm",
 *       "add" = "Drupal\shareholder_register\Form\ShareTransactionGroupForm",
 *       "edit" = "Drupal\shareholder_register\Form\ShareTransactionGroupForm",
 *       "delete" = "Drupal\shareholder_register\Form\ShareTransactionGroupDeleteForm",
 *     },
 *     "access" = "Drupal\shareholder_register\ShareTransactionGroupAccessControlHandler",
 *     "route_provider" = {
 *       "html" = "Drupal\shareholder_register\ShareTransactionGroupHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "share_transaction_group",
 *   admin_permission = "administer share transaction group entities",
 *   entity_keys = {
 *     "id" = "id",
 *     "bundle" = "type",
 *     "label" = "number",
 *     "uuid" = "uuid",
 *     "uid" = "user_id"
 *   },
 *   links = {
 *     "canonical" = "/admin/shareholder_register/share_transaction_group/{share_transaction_group}",
 *     "add-page" = "/admin/shareholder_register/share_transaction_group/add",
 *     "add-form" = "/admin/shareholder_register/share_transaction_group/add/{share_transaction_group_type}",
 *     "edit-form" = "/admin/shareholder_register/share_transaction_group/{share_transaction_group}/edit",
 *     "delete-form" = "/admin/shareholder_register/share_transaction_group/{share_transaction_group}/delete",
 *     "collection" = "/admin/shareholder_register/share_transaction/share_transaction_group",
 *   },
 *   bundle_entity_type = "share_transaction_group_type",
 *   field_ui_base_route = "entity.share_transaction_group_type.edit_form"
 * )
 */
class ShareTransactionGroup extends ContentEntityBase implements ShareTransactionGroupInterface {

  use EntityChangedTrait;

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage_controller, array &$values) {
    parent::preCreate($storage_controller, $values);
    $values += [
      'user_id' => \Drupal::currentUser()->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function delete() {
    if ($this->getName()) {
      drupal_set_message("You cannot delete a validated transaction!", 'error');
    }
    else {
      return parent::delete();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getName() {
    return $this->get('number')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setName($name) {
    $this->set('number', $name);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getDate() {
    return $this->get('date')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setDate($date) {
    return $this->set('date', $date);
  }

  /**
   * {@inheritdoc}
   */
  public function getNotes() {
    return $this->get('notes')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setNotes($notes) {
    return $this->set('notes', $notes);
  }

  /**
   * {@inheritdoc}
   */
  public function getState() {
    return $this->get('state')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setState($state) {
    return $this->set('state', $state);
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime($timestamp) {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getTransactions() {
    return $this->get('transactions')->referencedEntities();
  }

  /**
   * {@inheritdoc}
   */
  public function getShareholders() {
    $shareholders = [];
    foreach ($this->get('transactions')->referencedEntities() as $shareTransaction) {
      $shareholders[$shareTransaction->getShareholder()->id()] = $shareTransaction->getShareholder();
    }
    return $shareholders;
  }


  /**
   * {@inheritdoc}
   */
  public function getOwner() {
    return $this->get('user_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwnerId() {
    return $this->get('user_id')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwnerId($uid) {
    $this->set('user_id', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwner(UserInterface $account) {
    $this->set('user_id', $account->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getShareGroupType() {
    return ShareTransactionGroupType::load($this->bundle());
  }

  /**
   * {@inheritdoc}
   */
  public function getShareGroupTypeBaseType() {
    return $this->getShareGroupType()->getBaseType();
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['user_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Authored by'))
      ->setDescription(t('The user ID of author of the Share Transaction Group entity.'))
      ->setRevisionable(TRUE)
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setTranslatable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'author',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 5,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['number'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Name'))
      ->setDescription(t('The number of the Share Transaction Group entity.'))
      ->setSettings([
        'max_length' => 50,
        'text_processing' => 0,
        'wkf-editable' => ['state' => []],
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    $fields['date'] = BaseFieldDefinition::create('datetime')
      ->setLabel(t('Transaction Date'))
      ->setDescription(t('The Date of the Share Transaction Group.'))
      ->setSettings(array(
        'datetime_type' => 'date',
        'wkf-editable' => ['state' => []],
      ))
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['transactions'] = BaseFieldDefinition::create(
      'share_transaction_group_share_transactions')
      ->setLabel(t('Transactions'))
      ->setComputed(TRUE)
      ->setSettings([
        'target_type' => 'share_transaction',
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'weight' => -5,
      ]);

    $fields['notes'] = BaseFieldDefinition::create('text_long')
      ->setLabel(t('Notes'))
      ->setDescription(t('Notes for this transaction group'))
      ->setDisplayOptions('view', [
        'type' => 'text_default',
        'weight' => 10,
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'text_textfield',
        'weight' => 10,
      ]);

    $fields['state'] = BaseFieldDefinition::create('string')
      ->setLabel(t('State'))
      ->setDescription(t('State of the Share Transaction Group.'))
      ->setDefaultValue('draft');

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the entity was last edited.'));

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function actionValidate($date, $context = []) {
    $connection = \Drupal::database();
    $transaction = $connection->startTransaction();

    try {
      $this->internalActionValidate($date, $context);
    }
    catch (Exception $e) {
      $transaction->rollBack();
      throw $e;
    }

    // Notify everyone of new transaction.
    $event_dispatcher = \Drupal::service('event_dispatcher');
    foreach ($this->getTransactions() as $transaction) {
      $event = new ShareTransactionEvent($transaction);
      $event_dispatcher->dispatch(ShareTransactionEvent::VALIDATED_EVENT_NAME, $event);
    }
    $event = new ShareTransactionGroupEvent($this);
    $event_dispatcher->dispatch(ShareTransactionGroupEvent::VALIDATED_EVENT_NAME, $event);
  }

  /**
   * {@inheritdoc}
   */
  private function internalActionValidate($date, $context = []) {
    // Basic sanity checks.
    if (!count($this->getTransactions())) {
      throw new InvalidShareTransactionGroupException(
        "Cannot validate Share Transaction Group without transactions!");
    }

    $sum = 0;
    foreach ($this->getTransactions() as $transaction) {
      if ($transaction->getState() != "draft") {
        throw new InvalidShareTransactionGroupException(
          "Not all Transactions for the group are in state 'draft'!");
      }
      $sum += $transaction->getQuantity();
    }

    $type = ShareTransactionGroupType::load($this->bundle());
    if (count($this->getTransactions()) == 1) {
      if ($sum > 0 && $type->getBaseType() != 'issue') {
        throw new InvalidShareTransactionGroupException(
          "Positive total quantity but base type is not issue!");
      }
      elseif ($sum < 0 && $type->getBaseType() != 'redemption') {
        throw new InvalidShareTransactionGroupException(
          "Negative total quantity but base type is not 'redemption'!");
      }
      elseif ($sum == 0) {
        throw new InvalidShareTransactionGroupException(
          "Total quantity cannot be 0 if base type is not 'transfer'!");
      }
    }
    else {
      if ($sum != 0) {
        throw new InvalidShareTransactionGroupException(
          "Total quantity is not 0 but base type is 'transfer'!");
      }
    }

    // All is well, validate.
    if (!trim($this->getName())) {
      $connection = \Drupal::database();
      $result = $connection->query("select max(convert(coalesce(number, 0), SIGNED INTEGER)) + 1 as maxid from {share_transaction_group}");
      $row = $result->fetchObject();
      $this->setName($row->maxid);
    }

    $this->setState('valid');
    $this->setDate($date);
    $this->save();

    $transactions = $this->getTransactions();
    usort(
      $transactions,
      function ($a, $b) {
        return ($a->getQuantity() > $b->getQuantity()) ? +1 : -1;
      });

    $n = 1;
    $share_ids = [];
    foreach ($transactions as $transaction) {
      $transaction->preValidateTransaction($date);
      $transaction->getShareholder()->action_validate($date);

      if ($transaction->getQuantity() < 0) {
        $current_share_ids = $transaction->attachSharesToTransaction($date, []);
        $share_ids = array_merge($share_ids, $current_share_ids);

        // Validate share state.
        foreach (array_chunk($share_ids, 250) as $current_share_ids_chunk) {
          foreach (Share::loadMultiple($current_share_ids_chunk) as $share) {
            if ($share->getState() !== 'issued') {
              throw new ShareholderRegisterInvalidSharesException();
            }
          }
        }
      }
      else {
        $transaction_share_ids = array_splice($share_ids, 0, abs($transaction->getQuantity()));
        $current_share_ids = $transaction->attachSharesToTransaction($date, ['share_ids' => $transaction_share_ids]);
      }

      // Set transaction state, so Share presave has correct info.
      $transaction->setName("{$this->getName()}/{$n}");
      $transaction->setState('valid');
      $transaction->setDate($date);
      $transaction->save();

      if ($this->getShareGroupTypeBaseType() === 'issue') {
        // Allow to skip issuing for validation of large transactions.
        if (!array_key_exists('skip-issuing', $context) || !$context['skip-issuing']) {
          foreach (array_chunk($current_share_ids, 250) as $current_share_ids_chunk) {
            foreach (Share::loadMultiple($current_share_ids_chunk) as $share) {
              // Validate share state.
              if ($share->getState() !== 'draft') {
                throw new ShareholderRegisterInvalidSharesException();
              }
              $share->actionIssue();
            }
          }
        }
      }
      elseif ($this->getShareGroupTypeBaseType() === 'redemption') {
        foreach (array_chunk($current_share_ids, 250) as $current_share_ids_chunk) {
          foreach (Share::loadMultiple($current_share_ids_chunk) as $share) {
            // Validate share state.
            if ($share->getState() !== 'issued') {
              throw new ShareholderRegisterInvalidSharesException();
            }
            $share->actionRedeem();
          }
        }
      }
      else {
        foreach (array_chunk($current_share_ids, 250) as $current_share_ids_chunk) {
          foreach (Share::loadMultiple($current_share_ids_chunk) as $share) {
            // Update current holder.
            $share->save();
          }
        }
      }

      $shareholder = $transaction->getShareholder();
      $shareholder->setChangedTime(\Drupal::service('datetime.time')->getRequestTime());
      $shareholder->save();

      \Drupal::entityTypeManager()->getStorage('shareholder')->resetCache([
          $shareholder->id()
      ]);

      $n++;
    }
  }

}
