<?php

namespace Drupal\shareholder_register\Entity;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\user\UserInterface;

use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Defines the Share type entity.
 *
 * @ingroup shareholder_register
 *
 * @ContentEntityType(
 *   id = "share_type",
 *   label = @Translation("Share type"),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\shareholder_register\ShareTypeListBuilder",
 *     "views_data" = "Drupal\shareholder_register\Entity\ShareTypeViewsData",
 *     "translation" = "Drupal\shareholder_register\ShareTypeTranslationHandler",
 *
 *     "form" = {
 *       "default" = "Drupal\shareholder_register\Form\ShareTypeForm",
 *       "add" = "Drupal\shareholder_register\Form\ShareTypeForm",
 *       "edit" = "Drupal\shareholder_register\Form\ShareTypeForm",
 *       "delete" = "Drupal\shareholder_register\Form\ShareTypeDeleteForm",
 *     },
 *     "access" = "Drupal\shareholder_register\ShareTypeAccessControlHandler",
 *     "route_provider" = {
 *       "html" = "Drupal\shareholder_register\ShareTypeHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "share_type",
 *   admin_permission = "administer share type entities",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "name",
 *     "uuid" = "uuid",
 *     "uid" = "user_id",
 *     "langcode" = "langcode"
 *   },
 *   links = {
 *     "canonical" = "/admin/shareholder_register/share_type/{share_type}",
 *     "add-form" = "/admin/shareholder_register/share_type/add",
 *     "edit-form" = "/admin/shareholder_register/share_type/{share_type}/edit",
 *     "delete-form" = "/admin/shareholder_register/share_type/{share_type}/delete",
 *     "collection" = "/admin/shareholder_register/share_type",
 *   },
 *   field_ui_base_route = "share_type.settings"
 * )
 */
class ShareType extends ContentEntityBase implements ShareTypeInterface {

  use EntityChangedTrait;
  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage_controller, array &$values) {
    parent::preCreate($storage_controller, $values);
    $values += [
      'user_id' => \Drupal::currentUser()->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getName() {
    return $this->get('name')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setName($name) {
    $this->set('name', $name);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getParValue() {
    return $this->get('par_value')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getIssuePremium() {
    return $this->get('issue_premium')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getCurrentSharePrice() {
    return array_sum(
      array_map(function ($pc) {return $pc['amount']; },
        $this->getCurrentSharePriceComponents()));
  }

  /**
   * {@inheritdoc}
   */
  public function getCurrentSharePriceComponents() {
    $price_items = [
      'par_value' => [
        'label' => $this->t('Par Value'),
        'name' => 'par_value',
        'amount' => $this->get('par_value')->value,

      ],
      'issue_premium' => [
        'label' => $this->t('Issue Premium'),
        'name' => 'issue_premium',
        'amount' => $this->get('issue_premium')->value,

      ],
    ];
    $price_items = array_merge(
      $price_items,
      \Drupal::moduleHandler()->invokeAll('share_type_price', [$this])
    );

    \Drupal::moduleHandler()->alter('share_type_price', $this, $price_items);

    return $price_items;
  }

  /**
   * {@inheritdoc}
   */
  public function getShareCount() {
    return $this->get('share_count')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getShareTotalValue() {
    return $this->get('share_value')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime($timestamp) {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwner() {
    return $this->get('user_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwnerId() {
    return $this->get('user_id')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwnerId($uid) {
    $this->set('user_id', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwner(UserInterface $account) {
    $this->set('user_id', $account->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['user_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Authored by'))
      ->setDescription(t('The user ID of author of the Share type entity.'))
      ->setRevisionable(TRUE)
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setTranslatable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'author',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 5,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Name'))
      ->setDescription(t('The name of the Share type entity.'))
      ->setSettings([
        'max_length' => 50,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['par_value'] = BaseFieldDefinition::create('decimal')
      ->setLabel(t('Par Value'))
      ->setDescription(t('The Par Value of the Share type entity.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'number_decimal',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'number',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    // Deprecated in favor , should be removed.
    $fields['issue_premium'] = BaseFieldDefinition::create('decimal')
      ->setLabel(t('Current Issue Premium'))
      ->setDescription(t('The Current Issue Premium of the Share type entity.'))
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['share_count'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Shares'))
      ->setComputed(TRUE)
      ->setClass(\Drupal\shareholder_register\Plugin\Field\FieldType\ShareCountItemList::class)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'weight' => -5,
      ]);

    $fields['share_value'] = BaseFieldDefinition::create('decimal')
      ->setLabel(t('Share Value'))
      ->setComputed(TRUE)
      ->setClass(\Drupal\shareholder_register\Plugin\Field\FieldType\ShareTotalValueItemList::class)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'weight' => -5,
      ]);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the entity was last edited.'));

    return $fields;
  }

}
