<?php

namespace Drupal\shareholder_register\Entity;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\RevisionableContentEntityBase;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

use Drupal\Component\Utility\Html;

use Drupal\user\UserInterface;

use Drupal\shareholder_register\Event\ShareholderEvent;
use Drupal\shareholder_register\Exception\ShareholderRegisterInvalidShareholderIdentifierException;
use Drupal\shareholder_register\Plugin\Field\FieldType\AddressFieldType;

use Drupal\shareholder_register\Plugin\Field\FieldType\ShareTotalValueItemList;
use Drupal\shareholder_register\Plugin\Field\FieldType\ShareCountItemList;

/**
 * Defines the Shareholder entity.
 *
 * @ingroup shareholder_register
 *
 * @ContentEntityType(
 *   id = "shareholder",
 *   label = @Translation("Shareholder"),
 *   bundle_label = @Translation("Shareholder type"),
 *   handlers = {
 *     "storage" = "Drupal\shareholder_register\ShareholderStorage",
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\shareholder_register\ShareholderListBuilder",
 *     "views_data" = "Drupal\shareholder_register\Entity\ShareholderViewsData",
 *
 *     "form" = {
 *       "default" = "Drupal\shareholder_register\Form\ShareholderForm",
 *       "add" = "Drupal\shareholder_register\Form\ShareholderForm",
 *       "edit" = "Drupal\shareholder_register\Form\ShareholderForm",
 *       "delete" = "Drupal\shareholder_register\Form\ShareholderDeleteForm",
 *     },
 *     "access" = "Drupal\shareholder_register\ShareholderAccessControlHandler",
 *     "route_provider" = {
 *       "html" = "Drupal\shareholder_register\ShareholderHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "shareholder",
 *   revision_table = "shareholder_revision",
 *   revision_data_table = "shareholder_field_revision",
 *   revision_metadata_keys = {
 *     "revision_default" = "revision_default",
 *     "revision_user" = "revision_user",
 *     "revision_created" = "revision_created",
 *     "revision_log_message" = "revision_log_message",
 *   },
 *   admin_permission = "administer shareholder entities",
 *   entity_keys = {
 *     "id" = "id",
 *     "revision" = "vid",
 *     "bundle" = "type",
 *     "label" = "name",
 *     "uuid" = "uuid",
 *     "uid" = "user_id"
 *   },
 *   links = {
 *     "canonical" = "/admin/shareholder_register/shareholder/{shareholder}",
 *     "add-page" = "/admin/shareholder_register/shareholder/add",
 *     "add-form" = "/admin/shareholder_register/shareholder/add/{shareholder_type}",
 *     "edit-form" = "/admin/shareholder_register/shareholder/{shareholder}/edit",
 *     "delete-form" = "/admin/shareholder_register/shareholder/{shareholder}/delete",
 *     "version-history" = "/admin/shareholder_register/shareholder/{shareholder}/revisions",
 *     "revision" = "/admin/shareholder_register/shareholder/{shareholder}/revisions/{shareholder_revision}/view",
 *     "revision_revert" = "/admin/shareholder_register/shareholder/{shareholder}/revisions/{shareholder_revision}/revert",
 *     "revision_delete" = "/admin/shareholder_register/shareholder/{shareholder}/revisions/{shareholder_revision}/delete",
 *     "collection" = "/admin/shareholder_register/shareholder",
 *     "certificate" = "/admin/shareholder_register/shareholder/{shareholder}/certificate",
 *   },
 *   bundle_entity_type = "shareholder_type",
 *   field_ui_base_route = "entity.shareholder_type.edit_form"
 * )
 */
class Shareholder extends RevisionableContentEntityBase implements ShareholderInterface {

  use EntityChangedTrait;
  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function label() {
    if ($this->getState() == 'valid') {
      return $this->t("[@number] @name", [
        '@number' => $this->getNumber(),
        '@name' => $this->getName(),
      ]);
    }
    else {
      return $this->getName();
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage_controller, array &$values) {
    parent::preCreate($storage_controller, $values);
    $values += [
      'user_id' => \Drupal::currentUser()->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function delete() {
    if ($this->getNumber()) {
      \Drupal::service('messenger')->addError($this->t("You cannot delete a validated shareholder!"), 'error');
      return;
    }
    elseif (count($this->get('transactions')->referencedEntities())) {
      \Drupal::service('messenger')->addError($this->t("You cannot delete a shareholder for whom transactions exist!"), 'error');
      return;
    }
    else {
      return parent::delete();
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function urlRouteParameters($rel) {
    $uri_route_parameters = parent::urlRouteParameters($rel);

    if ($rel === 'revision_revert' && $this instanceof RevisionableInterface) {
      $uri_route_parameters[$this->getEntityTypeId() . '_revision'] = $this->getRevisionId();
    }
    elseif ($rel === 'revision_delete' && $this instanceof RevisionableInterface) {
      $uri_route_parameters[$this->getEntityTypeId() . '_revision'] = $this->getRevisionId();
    }

    return $uri_route_parameters;
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage) {
    parent::preSave($storage);

    foreach (array_keys($this->getTranslationLanguages()) as $langcode) {
      $translation = $this->getTranslation($langcode);

      // If no owner has been set explicitly, make the anonymous user the owner.
      if (!$translation->getOwner()) {
        $translation->setOwnerId(0);
      }
    }

    // If no revision author has been set explicitly, make the
    // shareholder owner the revision author.
    if (!$this->getRevisionUser()) {
      $this->setRevisionUserId($this->getOwnerId());
    }

    // Set current status.
    $this->set('current', (bool) $this->getShareCount());

    // Basic name and identifier mappings.
    $identifiers = $this->computeIdentifiers();
    $this->setName($identifiers['name']);

    // Set identifier only on non validated shareholders.
    if ($this->getState() == 'draft') {
      if (empty($this->getIdentifier())) {
        $this->setIdentifier($identifiers['identifier']);
      }
      elseif ($identifiers['identifier'] !== $this->getIdentifier()) {
        \Drupal::service('messenger')->addMessage(
          $this->t(
            "The identifier '@identifier' for shareholder '@name' is not up to date!",
            [
              '@identifier' => $this->getIdentifier(),
              '@name' => $identifiers['name'],
            ]
          )
        );
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function computeIdentifiers() {
    $identifier = mb_strtolower(trim($this->getAddress()->given_name)) . ' ' . mb_strtolower(trim($this->getAddress()->family_name));
    $name = trim($this->getAddress()->given_name) . ' ' . trim($this->getAddress()->family_name);
    $args = [
      'identifier' => $identifier,
      'name' => $name,
    ];
    \Drupal::moduleHandler()->alter('shareholder_identifier', $this, $args);
    return $args;
  }

  /**
   * {@inheritdoc}
   */
  public function validateIdentifier() {
    return \Drupal::service('shareholder_register.default')
      ->validateIdentifier($this->getIdentifier(), $this->id());
  }

  /**
   * {@inheritdoc}
   */
  public static function loadByNumber($name) {
    $query = \Drupal::entityQuery('shareholder')
      ->condition('number', $name)
      ->condition('state', 'valid');
    $values = $query->execute();
    if (count($values) == 1) {
      return Shareholder::load(reset($values));
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function getName() {
    return $this->get('name')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setName($name) {
    $this->set('name', $name);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getIdentifier() {
    return $this->get('identifier')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setIdentifier($identifier) {
    $this->set('identifier', $identifier);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getNumber() {
    return $this->get('number')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setNumber($number) {
    $this->set('number', $number);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getMail() {
    return $this->get('mail')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getAddress() {
    return $this->get('address');
  }

  /**
   * {@inheritdoc}
   */
  public function getFormattedAddress($options = []) {
    return AddressFieldType::formatAddress($this->get('address'), $options);
  }

  /**
   * {@inheritdoc}
   */
  public function getHTMLFormattedAddressOnly() {
    return nl2br(Html::escape($this->getFormattedAddress([
            'include_name' => FALSE,
    ])));
  }

  /**
   * {@inheritdoc}
   */
  public function getGivenName() {
    return $this->get('address')->given_name;
  }

  /**
   * {@inheritdoc}
   */
  public function getFamilyName() {
    return $this->get('address')->family_name;
  }

  /**
   * {@inheritdoc}
   */
  public function getAddressLine() {
    return $this->get('address')->address_line1;
  }

  /**
   * {@inheritdoc}
   */
  public function getFullAddressLine() {
    return implode(
      ' ',
      array_filter(
        array_map(
          'trim',
          [
            $this->get('address')->address_line1,
            $this->get('address')->premise,
            $this->get('address')->subpremise,
          ]
        )
      )
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getPostalCode() {
    return $this->get('address')->postal_code;
  }

  /**
   * {@inheritdoc}
   */
  public function getLocality() {
    return $this->get('address')->locality;
  }

  /**
   * {@inheritdoc}
   */
  public function getCountryCode() {
    return $this->get('address')->country_code;
  }

  /**
   * {@inheritdoc}
   */
  public function getCountry() {
    $country_manager = \Drupal::service('country_manager');
    $list = $country_manager->getList();
    return $list[$this->get('address')->country_code];
  }

  /**
   * {@inheritdoc}
   */
  public function getPreferredLangcode() {
    return $this->get('preferred_langcode')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getState() {
    return $this->get('state')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setState($state) {
    $this->set('state', $state);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getRegistrationDate() {
    return $this->get('registration_date')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setRegistrationDate($date) {
    $this->set('registration_date', $date);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getShareCount() {
    return $this->get('share_count')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getShareTotalValue() {
    return $this->get('share_value')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getShares() {
    return $this->getSharesAtDate();
  }

  /**
   * {@inheritdoc}
   */
  public function getShareIds() {
    return $this->getShareIdsAtDate();
  }

  /**
   * {@inheritdoc}
   */
  public function getTransactions() {
    return $this->get('transactions')->referencedEntities();
  }

  /**
   * {@inheritdoc}
   */
  public function getValidTransactions() {
    return array_filter(
      $this->get('transactions')->referencedEntities(),
      function ($t) {
        return $t->getState() == 'valid';
      }
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getDraftTransactions() {
    return array_filter(
      $this->get('transactions')->referencedEntities(),
      function ($t) {
        return $t->getState() == 'draft';
      }
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime($timestamp) {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwner() {
    return $this->get('user_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwnerId() {
    return $this->get('user_id')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwnerId($uid) {
    $this->set('user_id', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwner(UserInterface $account) {
    $this->set('user_id', $account->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['user_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Authored by'))
      ->setDescription(t('The user ID of author of the Shareholder entity.'))
      ->setRevisionable(TRUE)
      ->setSetting('wkf-editable', ['state' => ['draft']])
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setTranslatable(TRUE)
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Name'))
      ->setDescription(t('The name of the Shareholder entity.'))
      ->setRevisionable(TRUE)
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
        'wkf-editable' => ['state' => []],
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 1,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 1,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['number'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Shareholder Number'))
      ->setDescription(t('The Shareholder number.'))
      ->setRevisionable(TRUE)
      ->setSettings([
        'max_length' => 50,
        'text_processing' => 0,
        'wkf-editable' => ['state' => []],
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 2,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 2,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);


    $fields['identifier'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Identifier'))
      ->setDescription(t('The unique identifier of the Shareholder.'))
      ->setRevisionable(TRUE)
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
        'wkf-editable' => ['state' => ['draft']],
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 3,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 3,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['registration_date'] = BaseFieldDefinition::create('datetime')
      ->setLabel(t('Registration Date'))
      ->setDescription(t('The Date the Shareholder was registered.'))
      ->setRevisionable(TRUE)
      ->setSettings(array(
        'datetime_type' => 'date',
        'wkf-editable' => ['state' => ['draft']],
      ))
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['mail'] = BaseFieldDefinition::create('email')
      ->setLabel(t('Email'))
      ->setDescription(t('The email of the Shareholder.'))
      ->setRevisionable(TRUE)
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 5,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);


    $fields['address'] = BaseFieldDefinition::create('address_field_type')
      ->setLabel(t('Address'))
      ->setDescription(t('The Address of the Shareholder entity.'))
      ->setRevisionable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'address_default',
        'weight' => 6,
      ])
      ->setDisplayOptions('form', [
        'type' => 'address_default',
        'weight' => 6,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['preferred_langcode'] = BaseFieldDefinition::create('language')
      ->setLabel(t('Preferred language code'))
      ->setDescription(t("The user's preferred language code for receiving emails and viewing the site."))
      ->addPropertyConstraints('value', [
        'AllowedValues' => ['callback' => '\Drupal\user\Entity\User::getAllowedConfigurableLanguageCodes'],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);


    $fields['deceased_date'] = BaseFieldDefinition::create('datetime')
      ->setLabel(t('Date deceased'))
      ->setDescription(t('The Date the Shareholder has deceased.'))
      ->setRevisionable(TRUE)
      ->setSettings([
        'datetime_type' => 'date',
        'wkf-editable' => ['state' => ['valid']],
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 8,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 8,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['share_count'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Share Count'))
      ->setComputed(TRUE)
      ->setClass(ShareCountItemList::class)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'weight' => 9,
      ]);

    $fields['share_value'] = BaseFieldDefinition::create('decimal')
      ->setLabel(t('Share Value'))
      ->setComputed(TRUE)
      ->setClass(ShareTotalValueItemList::class)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'weight' => 10,
      ]);

    $fields['transactions'] = BaseFieldDefinition::create('shareholder_share_transactions')
      ->setLabel(t('Transactions'))
      ->setComputed(TRUE)
      ->setSettings([
        'target_type' => 'share_transaction',
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'share_transactions_formatter',
        'weight' => 11,
      ]);

    $fields['state'] = BaseFieldDefinition::create('string')
      ->setLabel(t('State'))
      ->setDescription(t('State of the Shareholder.'))
      ->setRevisionable(TRUE)
      ->setDefaultValue('draft');

    $fields['current'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Current holder of shares.'))
      ->setDescription(t('A boolean indicating whether the Shareholder is currently holding shares or has withdrawn.'))
      ->setRevisionable(TRUE)
      ->setDefaultValue(TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the entity was last edited.'));

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function action_validate($date) {
    if ($this->getState() == 'valid') {
      return $this;
    }

    if (!$this->validateIdentifier()) {
      $e = new ShareholderRegisterInvalidShareholderIdentifierException(
        "Invalid shareholder identifier ':identifier'!");
      $e->setMsgPlaceholders([
          ':identifier' => $this->computeIdentifiers()['identifier'],
      ]);
      throw $e;
    }

    $event = new ShareholderEvent($this);
    $event_dispatcher = \Drupal::service('event_dispatcher');

    $connection = \Drupal::database();
    $transaction = $connection->startTransaction();

    if (!$this->getNumber()) {
      $result = $connection->query("select max(convert(coalesce(number, 0), SIGNED INTEGER)) + 1 as maxid from {shareholder}");
      $row = $result->fetchObject();
      $this->setNumber($row->maxid);
    }

    $this->setState('valid');
    $this->setRegistrationDate($date);

    $event_dispatcher->dispatch(ShareholderEvent::PRE_VALIDATE_EVENT_NAME, $event);
    $this->save();
    $event_dispatcher->dispatch(ShareholderEvent::VALIDATED_EVENT_NAME, $event);

    \Drupal::service('messenger')->addStatus($this->t('New shareholder @name validated!', ['@name' => $this->getName()]));

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function actionRefuse($notes) {
    if ($this->getNumber()) {
      \Drupal::service('messenger')->addError(
        $this->t('Cannot refuse already validated Shareholder @name!', ['@name' => $this->getName()]));
      return $this;
    }

    $this->setState('refused');
    $this->save();

    $event = new ShareholderEvent($this);
    $event_dispatcher = \Drupal::service('event_dispatcher');
    $event_dispatcher->dispatch(ShareholderEvent::REFUSED_EVENT_NAME, $event);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function actionCancel() {
    if ($this->getNumber()) {
      \Drupal::service('messenger')->addError($this->t('Shareholder @name already registered!', ['@name' => $this->getName()]));
      return $this;
    }

    $this->setState('cancel');
    $this->save();

    $event = new ShareholderEvent($this);
    $event_dispatcher = \Drupal::service('event_dispatcher');
    $event_dispatcher->dispatch(ShareholderEvent::CANCELED_EVENT_NAME, $event);

    return $this;
  }

  /**
   * {@inheritdoc}
   *
   * @param \DateTime $date
   *   The last included date.
   * @param $end_transaction
   *   The last included transaction.
   */
  public function getShareIdsAtDate($date = NULL, $end_transaction = NULL, $use_payment_date = FALSE) {
    $s = \Drupal::service('shareholder_register.query');

    $share_ids = $s->getShareIdsAtDate(
      $this->id(),
      $date,
      $end_transaction,
      $use_payment_date
    );

    if (!$use_payment_date && count($share_ids) != $this->getShareCountAtDate($date, $end_transaction)) {
      throw new \Exception("Internal inconsistency in share count for shareholder id: {$this->id()} at date {$date}");
    }

    return $share_ids;
  }

  /**
   * {@inheritdoc}
   *
   * @param \DateTime $date
   *   The last included transaction.
   * @param string $end_transaction
   *   The last included transaction.
   */
  public function getSharesAtDate($date = NULL, string $end_transaction = NULL, $use_payment_date = FALSE) {
    return Share::loadMultiple($this->getShareIdsAtDate($date, $end_transaction, $use_payment_date));
  }

  /**
   * {@inheritdoc}
   *
   * @param \DateTime $date
   *   The last included transaction.
   * @param string $end_transaction
   *   The last included transaction.
   */
  public function getShareRevisionsAtDate($date = NULL, string $end_transaction = NULL, $use_payment_date = FALSE) {
    $service = \Drupal::service('shareholder_register.query');
    $share_ids = $this->getShareIdsAtDate($date, $end_transaction, $use_payment_date);
    return $service->getShareRevisionsAtDate($share_ids, $date);
  }

  /**
   * Retrieve list of shares held at date, which are still held today.
   *
   * @param \DateTime $date
   *   The last included transaction.
   * @param string $end_transaction
   *   The last included transaction.
   */
  public function getCurrentSharesAtDate($date = NULL, string $end_transaction = NULL, $use_payment_date = FALSE) {
    return Share::loadMultiple(
      $this->getCurrentShareIdsAtDate(
        $date,
        $end_transaction,
        $use_payment_date
      )
    );
  }

  /**
   * Retrieve list of share ids held at date, which are still held today.
   *
   * @param \DateTime $date
   *   The last included transaction.
   * @param string $end_transaction
   *   The last included transaction.
   */
  public function getCurrentShareIdsAtDate($date = NULL, string $end_transaction = NULL, $use_payment_date = FALSE) {
    return array_intersect(
      $this->getShareIdsAtDate(),
      $this->getShareIdsAtDate($date, $end_transaction, $use_payment_date)
    );
  }

  /**
   * Get list of transaction between begin and end inclusive.
   *
   * @param \DateTime $begin_date
   *   The begin date (inclusive).
   * @param \DateTime $end_date
   *   The end date (inclusive).
   * @param bool $use_payment_date
   *   Use payment date not validation date.
   */
  public function getValidTransactionsInRange($begin_date, $end_date, $use_payment_date = FALSE) {
    // TODO: review date condition.
    if ($begin_date instanceof \Datetime) {
      $begin_date = $begin_date->format('Y-m-d');
    }
    if ($end_date instanceof \Datetime) {
      $end_date = $end_date->format('Y-m-d');
    }

    // Do not use $this->get('transactions')->referencedEntities(), it is invalid during
    // transfer share group validation (because cached), eg when called in hook_shareholder_update.
    $source_transactions = ShareTransaction::loadMultiple(
      \Drupal::entityQuery('share_transaction')
      ->condition('shareholder_id', $this->id())
      ->sort('date', 'ASC')
      ->sort('name', 'ASC')
      ->execute()
    );

    $transactions = [];
    foreach ($source_transactions as $transaction) {
      if ($transaction->getState() != 'valid') {
        continue;
      }
      if ($begin_date && $transaction->getTransactionDate($use_payment_date) < $begin_date) {
        continue;
      }
      if ($end_date && $transaction->getTransactionDate($use_payment_date) > $end_date) {
        continue;
      }
      $transactions[] = $transaction;
    }
    usort($transactions, function ($a, $b) use ($use_payment_date) {
      if ($a->getTransactionDate($use_payment_date) == $b->getTransactionDate($use_payment_date)) {
        // TODO: coerce to int ?
        return ($a->getName() < $b->getName()) ? -1 : 1;
      }
      else {
        return ($a->getDate() < $b->getDate()) ? -1 : 1;
      }});
    return $transactions;
  }

  /**
   * {@inheritdoc}
   */
  public function getShareCountAtDate($date, $end_transaction = NULL) {
    if (!$date) {
      $connection = \Drupal::database();
      $result = $connection->query(
        "select max(date) as d from {share_transaction} where state = 'valid'")->fetchObject();
      $date = $result->d;
    }
    elseif ($date instanceof DateTime) {
      $date = $date->format('Y-m-d');
    }

    // Get share count with at date inclusive,
    // if $transaction is given, up to transaction number incl.
    $shares = 0;
    foreach ($this->getValidTransactionsInRange(FALSE, $date) as $transaction) {
      if ($transaction->getDate() > $date) {
        break;
      }
      // FIXME: intval on transaction might not work anymore with group/line naming.
      if ($end_transaction && $transaction->getDate() == $date && $end_transaction->getDate() == $date && intval($transaction->getName()) > intval($end_transaction->getName())) {
        break;
      }
      $shares += $transaction->quantity->value;
    }
    return $shares;
  }


  /**
   * {@inheritdoc}
   */
  public function sendExtractMail($context = NULL) {
    $email = $this->get('mail')->value;
    $ctx = [
      'shareholder' => $this,
      'entity' => $this,
    ];
    if (is_array($context)) {
      $ctx = $context + $ctx;
    }
    $params = [
      'context' => $ctx,
    ];
    $langcode = $this->getPreferredLangcode();
    $mailManager = \Drupal::service('plugin.manager.mail');
    $result = $mailManager->mail('shareholder_register', 'mail_shareholder_extract', $email, $langcode, $params, NULL, TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function getConversions() {
    $conversions = [];
    $query = \Drupal::service('shareholder_register.query');
    $conversion_dates = $query->getConversionDatesForShareholder($this->id());

    foreach ($conversion_dates as $conversion_date) {
      foreach ($query->getConversionSummaryAtDate($this->id(), $conversion_date) as $conversion) {
        $conversions[] = $conversion + [
          'date' => $conversion_date,
        ];
      }
    }
    return $conversions;
  }

  public function getGroupedShareIdsAtDate($date = NULL, $end_transaction = NULL, $use_payment_date = FALSE) {
    $share_ids = $this->getShareIdsAtDate($date, $end_transaction, $use_payment_date);
    return \Drupal::service('shareholder_register.default')->groupShareIdsByHash($share_ids);
  }
}
