<?php

namespace Drupal\shareholder_register\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;

use Drupal\shareholder_register\Entity\Share;
use Drupal\shareholder_register\Entity\Shareholder;
use Drupal\shareholder_register\ShareholderRegisterService;


/**
 * Class ConvertSharesForm.
 */
class ConvertSharesForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'convert_shares_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, Shareholder $shareholder = NULL) {
    $form['shares_at_date'] = [
      '#type' => 'shareholder_shares_at_date',
      '#weight' => '10',
      '#required' => TRUE,
    ];

    if ($shareholder) {
      $form['shares_at_date']['#default_value']['shareholder'] = $shareholder->id();
      $form['shares_at_date']['#default_value']['date'] = date('Y-m-d');
    }

    $weight = 40;
    $fields = \Drupal::service('entity_field.manager')->getFieldDefinitions('share', 'share');
    foreach (ShareholderRegisterService::getShareHashFields() as $hash_field) {
      if (!in_array($hash_field['type'], ['entity_reference', 'boolean'], TRUE)) {
        continue;
      }

      if ($hash_field['type'] === 'entity_reference') {
        $form[$hash_field['name']] = [
          '#type' => 'entity_autocomplete',
          '#title' => $hash_field['label'],
          '#weight' => $weight,
          '#target_type' => $fields[$hash_field['name']]->getSettings()['target_type'],
        ];
        if (isset($fields[$hash_field['name']]->getSettings()['handler_settings']['target_bundles'])) {
          $form[$hash_field['name']]['#selection_settings'] = [
            'target_bundles' => array_keys($fields[$hash_field['name']]->getSettings()['handler_settings']['target_bundles']),
          ];
        }
      }
      elseif ($hash_field['type'] === 'boolean') {
        $form[$hash_field['name']] = [
          '#type' => 'checkbox',
          '#title' => $hash_field['label'],
          '#weight' => $weight,
        ];
      }
      $weight += 10;
    }


    $form['revision_log_message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Revision log message'),
      '#weight' => $weight,
      '#required' => TRUE,
    ];

    $weight += 10;

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit'),
      '#weight' => $weight,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public static function updateShares(array $form, FormStateInterface $form_state) {
    return $form['shares'];
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $share_ids = $form_state->getValue('shares_at_date')['shares'];
    $service = \Drupal::service('shareholder_register.query');

    if (!count($share_ids)) {
      $form_state->setErrorByName(
        'shares_at_date][shares',
        $this->t('Select the shares to convert!')
      );
    }
    else {
      $invalid_shares = [];
      foreach ($share_ids as $share_id) {
        if ($service->getRevisionAtDate($share_id, $form_state->getValue('shares_at_date')['date']) !=
          $service->getRevisionAtDate($share_id, '9999-99-99')) {
          $invalid_shares[] = $share_id;
        }
      }
      if (count($invalid_shares)) {
        $form_state->setErrorByName(
          'shares_at_date][shares',
          $this->t('Shares @shares already have a conversion at a later date!', [
              '@shares' => implode(', ', $invalid_shares),
          ])
        );
      }
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $share_ids = $form_state->getValue('shares_at_date')['shares'];

    if (!count($share_ids)) {
      throw new \Exception('Internal error');
    }

    $shares = Share::loadMultiple($share_ids);
    $conversion_fields = [];
    foreach (ShareholderRegisterService::getShareHashFields() as $hash_field) {
      $conversion_fields[$hash_field['name']] = $form_state->getValue($hash_field['name']);
    }

    $updated_shares = ShareholderRegisterService::convertShares(
      $shares,
      $conversion_fields,
      $form_state->getValue('shares_at_date')['date'],
      $form_state->getValue('revision_log_message')
    );

    if (count($updated_shares)) {
      $this->messenger()->addStatus($this->t(
          'Conversion of shares :shares saved.',
          [
            ':shares' => \Drupal::service('shareholder_register.formatter')->sharesToRanges($updated_shares),
          ]
      ));
    }
  }

}
