<?php

namespace Drupal\shareholder_register\Form;

use Drupal\shareholder_register\Entity\Shareholder;
use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Form\FormStateInterface;

use Drupal\shareholder_register\Exception\ShareholderRegisterException;
use Drupal\shareholder_register\Exception\ShareTransactionNegativeShareCountException;
use Drupal\shareholder_register\Exception\ShareTransactionNegativeShareCountAtDateException;

/**
 * Form controller for Share transaction edit forms.
 *
 * @ingroup shareholder_register
 */
class ShareTransactionForm extends ContentEntityForm {

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, Shareholder $shareholder = NULL) {
    /* @var $entity \Drupal\shareholder_register\Entity\ShareTransaction */
    $form = parent::buildForm($form, $form_state);

    $form['share_type_id']['widget'][0]['target_id']['#required'] = TRUE;
    $form['share_value']['#access'] = FALSE;
    $form['shareholder_id']['widget'][0]['target_id']['#required'] = TRUE;
    if ($shareholder) {
      $form['shareholder_id']['widget'][0]['target_id']['#default_value'] = $shareholder;
    }
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Set transaction quantity before pre_validate.
    $cur_qty = $this->entity->getQuantity();
    $cur_shareholder = $this->entity->get('shareholder_id')->entity;

    $this->entity->setQuantity($form_state->getValues()['quantity']);
    $this->entity->set('shareholder_id', $form_state->getValue('shareholder_id'));

    try {
      $this->entity->preValidateTransaction($form_state->getValues()['date'][0]['value']);
    }
    catch (ShareTransactionNegativeShareCountException $e) {
      $form_state->setErrorByName('quantity', $this->t(
          'The total shares for the shareholder @shareholder would become negative', [
            '@shareholder' => $this->entity->getShareholder()->getName(),
          ])
      );
    }
    catch (ShareTransactionNegativeShareCountAtDateException $e) {
      $form_state->setErrorByName('quantity', $this->t(
        'The total shares for the shareholder @shareholder at @date would become negative',
        [
          '@date' => $form_state->get('date'),
          '@shareholder' => $this->entity->getShareholder()->getName(),
        ])
      );
    }
    catch (ShareholderRegisterException $e) {
      $this->messenger()->addMessage(
        $this->t(
          $e->getMessage(),
          $e->getMsgPlaceholders()),
        'status'
      );
    }
    catch (\Exception $e) {
      throw $e;
    }
    finally {
      $this->entity->setQuantity($cur_qty);
      $this->entity->set('shareholder_id', $cur_shareholder);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $entity = $this->entity;

    $status = parent::save($form, $form_state);

    switch ($status) {
      case SAVED_NEW:
        $this->messenger()->addMessage($this->t('Created the %label Share transaction.', [
          '%label' => $entity->label(),
        ]));
        break;

      default:
        $this->messenger()->addMessage($this->t('Saved the %label Share transaction.', [
          '%label' => $entity->label(),
        ]));
    }
    //$form_state->setRedirect('entity.share_transaction.canonical', ['share_transaction' => $entity->id()]);
    $form_state->setRedirect('view.share_transactions.page_2');
  }

}
