<?php

namespace Drupal\shareholder_register\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

use Drupal\shareholder_register\Entity\ShareTransaction;
use Drupal\shareholder_register\Exception\ShareholderRegisterException;

/**
 * Class ShareTransactionValidateForm.
 */
class ShareTransactionValidateForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'share_transaction_validate_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $current_url = Url::fromRoute('<current>')->toString();
    $params = Url::fromUserInput($current_url)->getRouteParameters();
    $form_state->set('transaction', $params['share_transaction']);

    $form['date'] = [
      '#type' => 'date',
      '#title' => $this->t('Date'),
      '#description' => $this->t('The date for the registration'),
      '#required' => TRUE,
    ];
    $form['date_confirmed'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Are you sure you wish to proceed with this date?'),
      '#access' => $form_state->get('needs_confirmation') ? TRUE : FALSE,
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $transaction = ShareTransaction::load($form_state->getStorage()['transaction']);
    if (($date = $form_state->getValue('date')) &&
      !$form_state->getValue('date_confirmed')) {
      // Move to confirmation page.
      if ($date > date('Y-m-d')) {
        $form_state->set('needs_confirmation', TRUE);
        $form_state->setRebuild(TRUE);
      }
      elseif (($payment_date = $transaction->getPaymentDate()) &&
        $date != $payment_date) {
        $form_state->set('needs_confirmation', TRUE);
        $form_state->setRebuild(TRUE);
      }
    }

    if (!$form_state->get('needs_confirmation') || $form_state->getValue('date_confirmed')) {
      try {
        $transaction->actionValidate($form_state->getValue('date'));
      }
      catch (ShareholderRegisterException $e) {
        \Drupal::messenger()->addError(
          $this->t('Could not validate transation:') . ' ' .
          $this->t($e->getMessage(), $e->getMsgPlaceholders()));
      }
    }
  }

}
