<?php

namespace Drupal\shareholder_register\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;

use Drupal\shareholder_register\Entity\Shareholder;
use Drupal\shareholder_register\Entity\ShareTransaction;
use Drupal\shareholder_register\Entity\ShareTransactionGroup;
use Drupal\shareholder_register\Entity\ShareTransactionGroupType;

use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Class TransferShareForm.
 */
class TransferShareForm extends FormBase {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'transfer_share_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, Shareholder $shareholder = NULL) {
    $options = [];
    foreach (ShareTransactionGroupType::loadMultiple() as $type) {
      if ($type->getBaseType() == 'transfer') {
        $options[$type->id()] = $type->label();
      }
    }

    $form['type'] = [
      '#type' => 'select',
      '#title' => $this->t('Type'),
      '#description' => $this->t('The type of transfer'),
      '#weight' => '10',
      '#options' => $options,
      '#required' => TRUE,
    ];
    $form['date'] = [
      '#type' => 'date',
      '#title' => $this->t('Date'),
      '#description' => $this->t('The date of the transfer'),
      '#weight' => '20',
      '#required' => TRUE,
      '#default_value' => date('Y-m-d'),
    ];
    $form['shareholder'] = [
      '#type' => 'entity_autocomplete',
      '#title' => $this->t('Shareholder'),
      '#description' => $this->t('The current holder of the shares to transfer'),
      '#weight' => '30',
      '#required' => TRUE,
      '#target_type' => 'shareholder',
    ];
    if ($shareholder) {
      $form['shareholder']['#default_value'] = $shareholder;
    }
    $form['recipient'] = [
      '#type' => 'entity_autocomplete',
      '#title' => $this->t('Recipient'),
      '#description' => $this->t('The new holder of the shares.'),
      '#weight' => '40',
      '#required' => TRUE,
      '#target_type' => 'shareholder',
    ];
    $form['number_of_shares'] = [
      '#type' => 'number',
      '#title' => $this->t('Number of shares'),
      '#description' => $this->t('Number of shares to transfer'),
      '#weight' => '50',
      '#min' => 0,
      '#required' => TRUE,
    ];
    $form['reason'] = [
      '#type' => 'text_format',
      '#title' => $this->t('Reason'),
      '#description' => $this->t('Reason for this transfer'),
      '#weight' => '60',
    ];
    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit'),
      '#weight' => '1000',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    if ($form_state->getValue('shareholder')) {
      $s = Shareholder::load($form_state->getValue('shareholder'));
      if ($s->getShareCount() < $form_state->getValue('number_of_shares')) {
        $form_state->setError($form['number_of_shares'], $this->t(
            "The shareholder has less than the requested number of shares!"
        ));
      }

      if (count($s->getSharesAtDate($form_state->getValue('date'))) < $form_state->getValue('number_of_shares')) {
        $form_state->setError($form['number_of_shares'], $this->t(
            "The shareholder has less than the requested number of shares at the specified date!"
        ));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $s = Shareholder::load($form_state->getValue('shareholder'));

    $selected_shares = [];
    if (!$selected_shares) {
      $shares = array_values($s->getCurrentSharesAtDate($form_state->getValue('date')));

      for ($i = 0; $i < $form_state->getValue('number_of_shares') && $shares[$i]; $i++) {
        $selected_shares[] = $shares[$i];
      }
    }

    $shares_by_characteristics = [];
    foreach ($selected_shares as $share) {
      $shares_by_characteristics[$share->getShareType()->id()][] = $share;
    }

    $group = ShareTransactionGroup::create([
      'type' => $form_state->getValue('type'),
    ]);
    $group->save();

    foreach ($shares_by_characteristics as $shares) {
      $t1 = ShareTransaction::create([
        'transaction_group' => $group->id(),
        'shareholder_id' => $form_state->getValue('shareholder'),
        'quantity' => -1 * count($shares),
        'share_type_id' => $shares[0]->getShareType()->id(),
        'notes' => $form_state->getValue('reason'),
      ]);
      $t1->save();

      $t2 = ShareTransaction::create([
        'transaction_group' => $group->id(),
        'shareholder_id' => $form_state->getValue('recipient'),
        'quantity' => count($shares),
        'share_type_id' => $shares[0]->getShareType()->id(),
        'notes' => $form_state->getValue('reason'),
      ]);
      $t2->save();
    }
    $group->actionValidate($form_state->getValue('date'), ['shares' => $selected_shares]);
  }

}
