<?php

namespace Drupal\shareholder_register\Plugin\Field\FieldType;

use Drupal\Component\Utility\Random;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinition;


/**
 * Plugin implementation of the 'address_field_type' field type.
 *
 * @FieldType(
 *   id = "address_field_type",
 *   label = @Translation("Shareholder Address"),
 *   description = @Translation("Shareholder Address"),
 *   default_widget = "shareholder_address_widget",
 *   default_formatter = "sharholder_address_formatter"
 * )
 */
class AddressFieldType extends FieldItemBase {

  public static function getFormat($options=NULL) {

    return [
      [
        [
          'name' => 'organization',
          'flex' => 12,
          'required' => (isset($options['organization']) && $options['organization']),
          'access' => (isset($options['organization']) && $options['organization']),
        ],
      ],
      [
        [
          'name' => 'given_name',
          'flex' => 6,
          'required' => (!isset($options['organization']) || !$options['organization']),
        ],
        [
          'name' => 'family_name',
          'flex' => 6,
          'required' => (!isset($options['organization']) || !$options['organization']),
        ],
      ],
      [
        [
          'name' => 'address_line1',
          'flex' => 8,
          'required' => TRUE,
        ],
        [
          'name' => 'premise',
          'flex' => 2,
          'access' => (isset($options['premise_and_subpremise']) && $options['premise_and_subpremise']),
        ],
        [
          'name' => 'subpremise',
          'flex' => 2,
          'access' => (isset($options['premise_and_subpremise']) && $options['premise_and_subpremise']),
        ],
      ],
      [
        [
          'name' => 'postal_code',
          'flex' => 2,
          'required' => TRUE,
        ],
        [
          'name' => 'locality',
          'flex' => 10,
          'required' => TRUE,
        ],
      ],
      [
        [
          'name' => 'country_code',
          'flex' => 4,
          'required' => TRUE,
          'type' => 'select',
          'options' => \Drupal::service('country_manager')->getList(),
          'default' => isset($options['default_country_code']) ? $options['default_country_code'] : '',
        ]
      ],
    ];
  }

  public static function getDefinition() {
    return [
      'given_name' => [
        'label' => t('Given name'),
        'type' => 'string',
        'size' => 25,
      ],
      'additional_name' => [
        'label' => t('Additional name'),
        'type' => 'string',
        'size' => 25,
      ],
      'family_name' => [
        'label' => t('Family name'),
        'type' => 'string',
        'size' => 25,
      ],
      'organization' => [
        'label' => t('Organization'),
        'type' => 'string',
        'size' => 60,
      ],
      'address_line1' => [
        'label' => t('First address line'),
        'type' => 'string',
        'size' => 60,
      ],
      'address_line2' => [
        'label' => t('Second address line'),
        'type' => 'string',
        'size' => 60,
      ],
      'premise' => [
        'label' => t('Premise'),
        'type' => 'string',
        'size' => 10,
      ],
      'subpremise' => [
        'label' => t('Subpremise'),
        'type' => 'string',
        'size' => 10,
      ],
      'postal_code' => [
        'label' => t('Postal code'),
        'type' => 'string',
        'size' => 10,
      ],
      'sorting_code' => [
        'label' => t('Sorting code'),
        'type' => 'string',
        'size' => 10,
      ],
      'dependent_locality' => [
        'label' => t('Dependent locality (i.e. neighbourhood)'),
        'type' => 'string',
        'size' => 30,
      ],
      'locality' => [
        'label' => t('Locality (i.e. city)'),
        'type' => 'string',
        'size' => 30,
      ],
      'administrative_area' => [
        'label' => t('Top-level administrative subdivision of the country'),
        'type' => 'string',
        'size' => 60,
      ],
      'country_code' => [
        'label' => t('Two-letter country code'),
        'type' => 'string',
        'size' => 60,
      ],
      'langcode' => [
        'label' => t('Language code'),
        'type' => 'string',
        'size' => 60,
      ],
    ];
  }


  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition) {
    // Prevent early t() calls by using the TranslatableMarkup.
    $properties = [];
    $properties['langcode'] = DataDefinition::create('string')
      ->setLabel(t('The language code.'));
    $properties['country_code'] = DataDefinition::create('string')
      ->setLabel(t('The two-letter country code.'));
    $properties['administrative_area'] = DataDefinition::create('string')
      ->setLabel(t('The top-level administrative subdivision of the country.'));
    $properties['locality'] = DataDefinition::create('string')
      ->setLabel(t('The locality (i.e. city).'));
    $properties['dependent_locality'] = DataDefinition::create('string')
      ->setLabel(t('The dependent locality (i.e. neighbourhood).'));
    $properties['postal_code'] = DataDefinition::create('string')
      ->setLabel(t('The postal code.'));
    $properties['sorting_code'] = DataDefinition::create('string')
      ->setLabel(t('The sorting code.'));
    $properties['address_line1'] = DataDefinition::create('string')
      ->setLabel(t('The first line of the address block.'));
    $properties['address_line2'] = DataDefinition::create('string')
      ->setLabel(t('The second line of the address block.'));
    $properties['premise'] = DataDefinition::create('string')
      ->setLabel(t('The premise.'));
    $properties['subpremise'] = DataDefinition::create('string')
      ->setLabel(t('The subpremise.'));
    $properties['organization'] = DataDefinition::create('string')
      ->setLabel(t('The organization'));
    $properties['given_name'] = DataDefinition::create('string')
      ->setLabel(t('The given name.'));
    $properties['additional_name'] = DataDefinition::create('string')
      ->setLabel(t('The additional name.'));
    $properties['family_name'] = DataDefinition::create('string')
      ->setLabel(t('The family name.'));

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition) {
    return [
      'columns' => [
        'langcode' => [
          'type' => 'varchar',
          'length' => 32,
        ],
        'country_code' => [
          'type' => 'varchar',
          'length' => 2,
        ],
        'administrative_area' => [
          'type' => 'varchar',
          'length' => 255,
        ],
        'locality' => [
          'type' => 'varchar',
          'length' => 255,
        ],
        'dependent_locality' => [
          'type' => 'varchar',
          'length' => 255,
        ],
        'postal_code' => [
          'type' => 'varchar',
          'length' => 255,
        ],
        'sorting_code' => [
          'type' => 'varchar',
          'length' => 255,
        ],
        'address_line1' => [
          'type' => 'varchar',
          'length' => 255,
        ],
        'address_line2' => [
          'type' => 'varchar',
          'length' => 255,
        ],
        'premise' => [
          'type' => 'varchar',
          'length' => 255,
        ],
        'subpremise' => [
          'type' => 'varchar',
          'length' => 255,
        ],
        'organization' => [
          'type' => 'varchar',
          'length' => 255,
        ],
        'given_name' => [
          'type' => 'varchar',
          'length' => 255,
        ],
        'additional_name' => [
          'type' => 'varchar',
          'length' => 255,
        ],
        'family_name' => [
          'type' => 'varchar',
          'length' => 255,
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getConstraints() {
    $constraints = parent::getConstraints();

    if ($max_length = $this->getSetting('max_length')) {
      $constraint_manager = \Drupal::typedDataManager()->getValidationConstraintManager();
      $constraints[] = $constraint_manager->create('ComplexData', [
        'family_name' => [
          'Length' => [
            'max' => $max_length,
            'maxMessage' => t('%name: may not be longer than @max characters.', [
              '%name' => $this->getFieldDefinition()->getLabel(),
              '@max' => $max_length
            ]),
          ],
        ],
      ]);
    }

    return $constraints;
  }

  /**
   * {@inheritdoc}
   */
  public function storageSettingsForm(array &$form, FormStateInterface $form_state, $has_data) {
    $elements = [];

    $elements['max_length'] = [
      '#type' => 'number',
      '#title' => t('Maximum length'),
      '#default_value' => $this->getSetting('max_length'),
      '#required' => TRUE,
      '#description' => t('The maximum length of the field in characters.'),
      '#min' => 1,
      '#disabled' => $has_data,
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty() {
    $value = $this->get('family_name')->getValue().$this->get('given_name')->getValue().$this->get('organization')->getValue().$this->get('country_code')->getValue();
    return $value === NULL || $value === '';
  }

  /**
   * {@inheritdoc}
   */
  public static function formatAddress($address, $options=[]) {
    $r = [];
    foreach (AddressFieldType::getFormat() as $line_index => $format_line) {
      $line = '';
      foreach ($format_line as $property_dict) {
        $property = $property_dict['name'];

        if (isset($options['include_name']) && !$options['include_name'] && in_array($property, ['given_name', 'family_name', 'organization'])) {
          continue;
        }

        if ($property == 'country_code') {
          if ($address->country_code) {
            $line .= \Drupal::service('country_manager')->getList()[$address->country_code]->render();
          }
        } else {
          $line .= "{$address->{$property}} ";
        }
      }
      if (trim($line)) {
        $r[] = trim($line);
      }
    }
    return implode("\n", array_filter($r));
  }
}
