<?php
namespace Drupal\shareholder_register\Plugin\ShareholderRegisterExportPlugin;

use Drupal\Core\Link;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;

use Drupal\shareholder_register\Entity\Share;
use Drupal\shareholder_register\Plugin\ShareholderRegisterExportPluginInterface;
use Drupal\shareholder_register\ShareholderRegisterService;
use Drupal\shareholder_register\ShareholderRegisterExportService;
use Drupal\shareholder_register\ShareholderSharesLine;

/**
 * Export Register At Date.
 *
 * @ShareholderRegisterExportPlugin(
 *   id = "register_at_date_by_share_hash",
 *   label = @Translation("Register At Date (per share hash)"),
 *  )
 */
class RegisterAtDateByShareHash extends PluginBase implements PluginFormInterface, ShareholderRegisterExportPluginInterface {
  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {

    $form['date'] = [
      '#type' => 'date',
      '#title' => $this->t('Date'),
      '#required' => TRUE,
      '#weight' => '0',
    ];
    $form['use_payment_date'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use date of payment'),
      '#default_value' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['form_values'] = $form_state->getValues();
  }

  /**
   * {@inheritdoc}
   */
  public function getExportBatch() {
    $config = [
      'date' => $this->configuration['form_values']['date'],
      'use_payment_date' => $this->configuration['form_values']['use_payment_date'],
      'compute_ids_callback' => 'Drupal\shareholder_register\ShareholderRegisterExportService::allValidShareholders',
      'compute_callback' => 'Drupal\shareholder_register\Plugin\ShareholderRegisterExportPlugin\RegisterAtDateByShareHash::shareholderAtDateByShareHash',
      'write_callback' => 'Drupal\shareholder_register\Plugin\ShareholderRegisterExportPlugin\RegisterAtDateByShareHash::writeShareholderAtDateByShareHash',
      'page_title' => $this->t(
        'Shareholder Register state at date @date (@type)',
        [
          '@date' => $this->configuration['form_values']['date'],
          '@type' => $this->configuration['form_values']['use_payment_date'] ? $this->t('Date of payment') : $this->t('Date of validation'),
        ]),
      'column_headers' => [
        $this->t('Shareholder Number'),
        $this->t('Shareholder Name'),
        $this->t('Email'),
        $this->t('Registration Date'),
        $this->t('Street'),
        $this->t('Housenumber'),
        $this->t('Box'),
        $this->t('Postal code'),
        $this->t('Locality'),
        $this->t('Country'),
        $this->t('Shares'),
        $this->t('Number of shares'),
      ],
    ];

    foreach (ShareholderRegisterService::getShareHashFields() as $field_data) {
      $config['column_headers'][] = $field_data['label'];
    }

    $this->shareholderRegisterExport = \Drupal::service('shareholder_register.export');
    $batch = $this->shareholderRegisterExport->getExportBatch($config);

    return $batch;
  }

  /**
   * {@inheritdoc}
   */
  public static function shareholderAtDateByShareHash($config, $shareholder, &$context) {
    ShareholderRegisterExportService::shareholderAtDate($config, $shareholder, $context);
    $data = $context['results']['totals'][$shareholder->id()];
    unset($context['results']['totals'][$shareholder->id()]);

    if (!isset($data['share_ids'])) {
      return;
    }

    $formatter = \Drupal::service('shareholder_register.formatter');
    $query = \Drupal::service('shareholder_register.query');

    $grouped = [];
    foreach ($query->getShareRevisionsAtDate($data['share_ids'], $config['date']) as $share) {
      $hash = ShareholderRegisterService::getShareHash($share);
      if (!isset($grouped[$hash])) {
        $grouped[$hash] = [];
      }
      $grouped[$hash][] = $share->id();
    }

    foreach ($grouped as $hash => $share_ids) {
      $context['results']['totals'][] = [
        'shareholder_id' => $shareholder->id(),
        'share_ids' => $share_ids,
        'shares' => $formatter->shareIdsToRanges($share_ids),
        'share_hash' => $hash,
        'count' => count($share_ids),
      ];
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function writeShareholderAtDateByShareHash($config, $group, $sheet, $row, &$context) {
    ShareholderRegisterExportService::writeShareholderAtDate($config, $group, $sheet, $row, $context);
    $col = 'M';
    foreach (ShareholderRegisterService::getShareHashTextComponents($group['share_hash']) as $component) {
      $sheet->setCellValue("{$col}{$row}", $component);
      $col = chr(ord($col) + 1);
    }
  }
}
